function ir = var_ir(dat,p,hmax) 

% Creates h-step impulse response functions from estimated VAR(p) model with intercept

% Inputs:
%	dat 				nxm data matrix
%	p				VAR order p >= 1
%	hmax				horizon, hmax >= 1

% Output:
%	ir	(hmax+1) x m x m 	impulse responses
%					ir(h,j,i) is the impulse response of y_j with respect to e_i at horizon h
%					where e are the orthogonalized shocks.
%	The shocks are orthogonalized using a lower triangular Cholesky decomposition.	

n = size(dat,1)-p;
m = size(dat,2);
k = m*p+1;
y = dat(1+p:n+p,:);
x = ones(n,k);
for i = 1:p
    x(:,(i-1)*m+1:i*m) = dat(1+p-i:n+p-i,:);
end

B = ((x'*x)\x'*y)';
e = y - x*B';
sigma = (e'*e) / (n-k);
H = chol(sigma,'lower');

J = [eye(m*(p-1)), zeros(m*(p-1),m+1); zeros(1,m*p), 1];
P = [B(:,1:m*p), zeros(m,1); J];

ir = zeros(hmax+1,m,m);
Ph = eye(k);
ir(1,:,:) = H;
for h = 1:hmax
  Ph = P*Ph;
  ir(h+1,:,:) = Ph(1:m,1:m)*H;
end

