#########################################################################
##  PROC MAIN
##  
##  This file contains the R procedure MAIN
##  
##      written by
##  
##  Bruce E. Hansen
##  Department of Economics
##  Social Science Building
##  University of Wisconsin
##  Madison, WI 53706-1393
##  behansen@wisc.edu
##  http://www.ssc.wisc.edu/~bhansen/
##    
##  *************************************************************
## 
##  FORMAT: out <- main(y,x,model,choice,k)
##  INPUT:  y - depend variable
##          x - data matrix for independent variables 
##              (first row is first observation)
##              
##          model - choice for model =2  C
##                                   =3  C/T
##                                   =4  C/S
##          choice - only in ADF test, =1  pre-specified AR lag
##                                     =2  AIC-chosen AR lag
##                                     =3  BIC-chosen AR lag
##                                     =4  downward-t-chosen AR lag
##          k - maximum lag for ADF test
##  OUTPUT: print automatically 
##          Za*, breakpoint for Za*, 
##          Zt*, breakpoint for Zt*,
##          ADF*, breakpoint for ADF* and AR lag chosen for ADF*
##  GLOBAL VARIABLES: none
##  EXTERNAL PROCEDURES: adf,  phillips
##  NB: Constant included in regression
##
#########################################################################

#****************  Main procedure *******************#
main <- function(y,x,model,choice,k){
   n <- nrow(y)
   begin <- round(0.15*n)
   final <- round(0.85*n)
   temp1 <- matrix(0,(final-begin+1),1)
   temp2 <- temp1
   temp3 <- temp1
   temp4 <- temp1
   for (t in begin:final){
     dummy <- rbind(matrix(0,t,1),matrix(1,n-t,1)) 
     # adjust regressors for different models #
     if (model==3) x1 <- cbind(matrix(1,n,1),dummy,as.matrix(seq(1,n,1)),x)
     if (model==4) x1 <- cbind(matrix(1,n,1),dummy,x,(dummy%*%matrix(1,1,ncol(x)))*x)
     if (model==2) x1 <- cbind(matrix(1,n,1),dummy,x)

     # computer ADF for each t  #
     out <- adf(y,x1,k,choice)
     temp1[t-begin+1] <- out$stat
     temp2[t-begin+1] <- out$lag
   
     # compute Za or Zt for each t  #
     out <- phillips(y,x1)
     temp3[t-begin+1] <- out$za
     temp4[t-begin+1] <- out$zt
   }

   #  ADF test #
   tstat <- min(temp1)
   lag <- which.min(temp1)
   breakpta <- (lag+begin-1)/n
   lag <- temp2[lag]
   cat ("******** ADF Test ***********" ,"\n")
   cat ("t-statistic = ", tstat,"\n")
   cat ("AR lag = ", lag,"\n")
   cat ("break point(ADF) = ", breakpta,"\n")
   cat ("\n")

   #  Phillips test #
   za <- min(temp3)
   breakpt1 <- (which.min(temp3)+begin-1)/n
   zt <- min(temp4)
   breakpt2 <- (which.min(temp4)+begin-1)/n
   cat ("******** Phillips Test ********","\n")
   cat ("Zt =              ", zt,"\n")
   cat ("breakpoint(Zt) =  ", breakpt2,"\n")
   cat ("Za =              ", za,"\n")
   cat ("breakpoint(Za) =  ", breakpt1,"\n")
   cat ("\n")
}
# -------------------------------------------------------------- #


##**********************  PROC ADF  *****************************
##   FORMAT
##          out <- adf(y,x,kmax,choice){ 
##          out$stat
##          out$lag
##   INPUT
##          y - dependent variable
##          x - independent variables
##          kmax - maximum lag for ADF test
##          choice - only in ADF test, =1  pre-specified AR lag
##                                     =2  AIC-chosen AR lag
##                                     =3  BIC-chosen AR lag
##                                     =4  downward-t-chosen AR lag
##   OUTPUT
##          stata - ADF statistic
##          lag - the lag length
##   GLOBAL VARIABLES: none
##   EXTERNAL PROCEDURES: estimate
##****************************************************************

#************** ADF for each breakpoint ********************#

adf <- function(y,x,kmax,choice){
   # compute ADF  #
   n <- nrow(y)
   out <- estimate(y,x)
   e <- out$e
   de <- as.matrix(e[2:n]-e[1:(n-1)]) # difference of residuals #

   ic <- 0
   k <- kmax
   temp1 <- matrix(0,kmax+1,1)
   temp2 <- matrix(0,kmax+1,1)
   while (k>=0){
      yde <- as.matrix(de[(1+k):(n-1)])
      n1 <- nrow(yde)
      #  set up matrix for independent variable(lagged residuals)  #
      xe <- as.matrix(e[(k+1):(n-1)])
      j <- 1
      while (j <= k){
         xe <- cbind(xe,de[(k+1-j):(n-1-j)])
         j <- j+1
      }
      out <- estimate(yde,xe)
      b <- out$b
      e1 <- out$e
      sig2 <- out$sig2
      se <- out$se 
      if (choice==1){  # K is pre-specified #
          temp1[k+1] <- -1000   # set an random negative constant #
          temp2[k+1] <- b[1]/se[1]
          break}
      if (choice==2){  # K is determined by AIC #
         aic <- log(t(e1)%*%e1/n1)+2*(k+2)/n1
         ic <- aic}
      if (choice==3){  # K is determined by BIC #
         bic <- log(t(e1)%*%e1/n1)+(k+2)*log(n1)/n1
         ic <- bic}
      if (choice==4){  # K is determined by downward t #
         if (abs(b[k+1]/se[k+1])>= 1.96 | k==0){
            temp1[k+1] <- -1000    # set an random negative constant #
            temp2[k+1] <- b[1]/se[1]
            break
         }
      }
      temp1[k+1] <- ic
      temp2[k+1] <- b[1]/se[1]
      k <- k-1
   }

   lag <- which.min(temp1)
   tstat <- temp2[lag]
   list(stata=tstat,lag=lag-1)
}
# ------------------------------------------------------------ #



##*********************  PROC PHILLIPS  *****************************
##   FORMAT
##          out <- phillips(y,x)
##          out$za
##          out$zt  
##   INPUT
##          y - dependent variable
##          x - independent variables
##   OUTPUT
##          za - the Phillips test statistic
##          zt - the Phillips test statistic
##   GLOBAL VARIABLES: none
##*******************************************************************

#************** Za or Zt for each breakpoint **************#

phillips <- function(y,x){
   n=nrow(y)

   # OLS regression #
   b <- qr.solve(x,y)
   e <- y-x%*%b

   # OLS regression on residuals #
   be <- qr.solve(e[1:(n-1)],e[2:n])
   ue <- as.matrix(e[2:n])-as.matrix(e[1:(n-1)])%*%be

   # calculate bandwidth number #
   nu <- nrow(ue)
   bu <- qr.solve(ue[1:(nu-1)],ue[2:nu])
   uu <- as.matrix(ue[2:nu])-as.matrix(ue[1:(nu-1)])%*%bu
   su <- mean(uu^2)
   a2 <- (4*bu^2*su/(1-bu)^8)/(su/(1-bu)^4)
   bandwidth <- 1.3221*((a2*nu)^0.2)

   m <- bandwidth
   j <- 1
   lemda <- 0
   while (j<=m){
      gama <- t(as.matrix(ue[1:(nu-j)]))%*%as.matrix(ue[(j+1):nu])/nu
      c <- j/m
      w <- (75/(6*pi*c)^2)*(sin(1.2*pi*c)/(1.2*pi*c)-cos(1.2*pi*c))
      lemda <- lemda+w*gama
      j <- j+1
   }

   # calculate Za and Zt for each t #
   p <- sum(e[1:(n-1)]*e[2:n]-lemda)/sum(e[1:(n-1)]^2)
   za <- n*(p-1)
   sigma2 <- 2*lemda+t(ue)%*%ue/nu
   s <- sigma2/(t(as.matrix(e[1:(n-1)]))%*%as.matrix(e[1:(n-1)]))
   zt <- (p-1)/sqrt(s)
   list(za=za,zt=zt)
}
# ------------------------------------------------------------ #


##**********************  PROC ESTIMATE  *****************************
##   FORMAT
##          { b,e,sig2,se } = estimate(y,x)
##   INPUT
##        y  - dependent variable
##        x - independent variables
##   OUTPUT
##  b - OLS estimates
##  e - residuals
##  sig2 - variance
##  se - standard error for coefficients
##   GLOBAL VARIABLES: none
##********************************************************************
## *****  ols regression ****** ## 
estimate <- function(y,x){
   m <- solve(t(x)%*%x) 
   b <- m%*%(t(x)%*%y)
   e <- y-x%*%b
   sig2 <- (t(e)%*%e)/(nrow(y)-ncol(x))
   se <- sqrt(diag(m)*sig2)
   list(b=b,e=e,sig2=sig2,se=se)
}
# ---------------------------------------------------------------- #


