% SHIFTS.M

% This is a Matlab program.
% It calculates the statistics discussed in
% "Residual-based tests for cointegration in models
% with regime shifts"
% by Allan W. Gregory and Bruce E. Hansen

% Questions about the program can be addressed to

% Bruce E. Hansen
% Department of Economics
% Social Science Building
% University of Wisconsin
% Madison, WI 53706-1393
% bhansen@ssc.wisc.edu
% http://www.ssc.wisc.edu/~bhansen/


%* ******** testing program ******** *%
function shifts()
load mon59m.txt;xx1=mon59m;
load mon46q.txt;xx2=mon46q;
load mon47q.txt;xx3=mon47q;
load mon47m.txt;xx4=mon47m;
global out;
out=fopen('shiftsout.txt','wt');

% ----- Create Quarterly M1 and 3M-TB Series from Monthly Series ----- %
xx1q=zeros(4*(length(xx1)-5)/12,1);
xx4q=zeros(length(xx3),1);
jm=1;
jmc=1;
endm=length(xx1)-5;
while jm<=endm
    xx1q(jmc,:)=mean(xx1(jm:jm+2,:))';
    jmc=jmc+1;
    jm=jm+3;
end;
jr=3;
jrc=1;
endr=length(xx4(:,1))-5;

while jr<=endr
    xx4q(jrc,:)=xx4(jr,2);
    jrc=jrc+1;
    jr=jr+3;
end;

% ----- Set Sample: 1959.1 - 1990.4 ----- %
ynr=xx3(49:176,1); % ynr - Net National Product, Billions of 82$ %
pp=xx2(53:180,1)./xx3(49:176,1); % pp - Implicit Price Deflator, PI %
m1=xx1q(1:128,1)./pp; % m1/p - M1, Billions of $ %
sr=xx4q(49:176,1); % sr - 3 Month T-Bills %

obs=length(m1(:,1));

n=obs;
md=log(m1);
ynr=log(ynr);

%"Log of Real Money Balances";y;
%"Income and Interest Rates";x;
y=md;
x=[ynr,sr];
main(y,x,4,4,6);

%*************************************************************************
%----PROC MAIN
%----FORMAT: call  main(y,x,model,choice,k)
%----INPUT:      y - depend variable
%        x - data matrix for independent variables (first row is
%first observation)
%                model - choice for model        =2  C
%                        =3  C/T
%                        =4  C/S
%        choice - only in ADF test,  =1  pre-specified AR lag
%                        =2  AIC-chosen AR lag
%                        =3  BIC-chosen AR lag
%                        =4  downward-t-chosen AR lag
%        k - maximum lag for ADF test
%----OUTPUT: print automatically Za*, breakpoint for Za*, Zt*, breakpoint for Zt*
%, ADF*,                        breakpoint for ADF* and AR lag chosen for ADF*
%----GLOBAL VARIABLES: none
%----EXTERNAL PROCEDURES: adf,  phillips
%----NB: Constant included in regression
%************************************************************************/

%*
%****************  Main procedure *******************
%*

function main(y,x,model,choice,k);
global out;
n=length(y(:,1));
begin=round(.15*n);
final=round(.85*n);
temp1=zeros(final-begin+1,1);
temp2=temp1;
temp3=temp1;
temp4=temp1;
t=begin;
while t<=final
    dummy=[zeros(t,1);ones(n-t,1)];
    % adjust regressors for different models %
    if model==3;
        x1=[ones(n,1),dummy,(1:n)',x];
    elseif model==4
        x1=[ones(n,1),dummy,x,(dummy*ones(1,length(x(1,:)))).*x];
    elseif model==2
        x1=[ones(n,1),dummy,x];
    end;
    % computer ADF for each t %
    [temp1(t-begin+1),temp2(t-begin+1)]=adf(y,x1,k,choice);
    % compute Za or Zt for each t %
    [temp3(t-begin+1),temp4(t-begin+1)]=phillips(y,x1);
    t=t+1;
end;

% ADF test %
[tstat,lag]=min(temp1);
breakpta=(lag+begin-1)/n;
lag=temp2(lag);
fprintf(out,'******** ADF Test ***********\n');
fprintf(out,'t-statistic = %f\n',tstat);
fprintf(out,'AR lag= %u\n',lag);
fprintf(out,'Break point(ADF)=%f\n\n',breakpta);

% Phillips test %
[za,mindtemp3]=min(temp3);
breakpt1=(mindtemp3+begin-1)/n;
[zt,mindtemp4]=min(temp4);
breakpt2=(mindtemp4+begin-1)/n;
fprintf(out,'******** Phillips Test ********\n');
fprintf(out,'Zt=  %f\n',zt)
fprintf(out,'breakpoint(Zt) = %f\n',breakpt2);
fprintf(out,'Za=  %f\n',za);
fprintf(out,'breakpoint(Za) = %f\n\n',breakpt1);

%**********************  PROC ADF  *****************************
%**   FORMAT
%**          { stat,lag } = adf(y,x)
%**   INPUT
%**        y - dependent variable
%**        x - independent variables
%**   OUTPUT
%**  stata - ADF statistic
%**  lag - the lag length
%**   GLOBAL VARIABLES: none
%**   EXTERNAL PROCEDURES: estimate
%**********************************************************************/

%*************** ADF for each breakpoint ********************
function [tstat,lag1]=adf(y,x,kmax,choice);
% compute ADF %
n=length(y(:,1));


[b,e,sig2,se]=estimate(y,x);
de=e(2:n)-e(1:n-1); % difference of residuals %

ic=0;
k=kmax;
temp1=zeros(kmax+1,1);
temp2=zeros(kmax+1,1);
while k>=0;
    yde=de(1+k:n-1);
    n1=length(yde(:,1));
    % set up matrix for independent variable(lagged residuals) %
    xe=e(k+1:n-1);
    j=1;
    while j<=k
        xe=[xe,de(k+1-j:n-1-j)];
        j=j+1;
    end;
    [b,e1,sig2,se]=estimate(yde,xe);
    if choice==1; % K is pre-specified %
        temp1(k+1)=-1000; % set an random negative constant %
        temp2(k+1)=b(1)/se(1);
        break;
    elseif choice==2; % K is determined by AIC %
        aic=log(e1'*e1/n1)+2*(k+2)/n1;
        ic=aic;
    elseif choice==3; % K is determined by BIC %
        bic=log(e1'*e1/n1)+(k+2)*log(n1)/n1;
        ic=bic;
    elseif choice==4; % K is determined by downward t %
        if (abs(b(k+1)/se(k+1))>=1.96|k==0)
            temp1(k+1)=-1000; % set an random negative constant %
            temp2(k+1)=b(1)/se(1);
            break;
        end;
    end;
    temp1(k+1)=ic;
    temp2(k+1)=b(1)/se(1);
    k=k-1;
end;
[ttee,lag]=min(temp1);
tstat=temp2(lag);
lag1=lag-1;        
 
%**********************  PROC PHILLIPS  *****************************
%**   FORMAT
%**  { za,zt } = phillips(y,x)
%**   INPUT
%**  y  - dependent variable
%**  x - independent variables
%**   OUTPUT
%**  za - the Phillips test statistic
%**  zt -  the Phillips test statistic
%**   GLOBAL VARIABLES: none
%**********************************************************************/

%*************** Za or Zt for each breakpoint ********************

function [za,zt]=phillips(y,x);
n=length(y(:,1));

% OLS regression %
b=(y'/x')';
e=y-x*b;

% OLS regression on residuals %
be=(e(2:n)'/e(1:n-1)')';
ue=e(2:n)-e(1:n-1)*be;

% calculate bandwidth number %
nu=length(ue(:,1));
bu=(ue(2:nu)'/ue(1:nu-1)')';
uu=ue(2:nu)-ue(1:nu-1)*bu;
su=mean(uu.^2)';
a2=(4*bu^2*su/(1-bu)^8)/(su/(1-bu)^4);
bandwidth=1.3221*((a2*nu)^0.2);

m=bandwidth;
j=1;
lemda=0;
while j<=m
    gama=ue(1:nu-j)'*ue(j+1:nu)/nu;
    c=j/m;
    w=(75/(6*pi*c)^2)*(sin(1.2*pi*c)/(1.2*pi*c)-cos(1.2*pi*c));
    lemda=lemda+w*gama;
    j=j+1;
end;

% calculate Za and Zt for each t %
p=sum(e(1:n-1).*e(2:n)-lemda)/sum(e(1:n-1).^2);
za=n*(p-1);
sigma2=2*lemda+ue'*ue/nu;
s=sigma2/(e(1:n-1)'*e(1:n-1));
zt=(p-1)/sqrt(s);

%**********************  PROC ESTIMATE  *****************************
%**   FORMAT
%**          { b,e,sig2,se } = estimate(y,x)
%**   INPUT
%**        y  - dependent variable
%**        x - independent variables
%**   OUTPUT
%**  b - OLS estimates
%**  e - residuals
%**  sig2 - variance
%**  se - standard error for coefficients
%**   GLOBAL VARIABLES: none
%**********************************************************************/
% * *****  ols regression ****** */
function [b,e,sig2,se]=estimate(y,x);
m=inv(x'*x);
b=m*(x'*y);
e=y-x*b;
sig2=(e'*e)/(length(y(:,1))-length(x(1,:)));
se=sqrt(diag(m)*sig2);
