/* SHIFTS.PRG

This is a GAUSS program.
It calculates the statistics discussed in
"Residual-based tests for cointegration in models
with regime shifts"
by Allan W. Gregory and Bruce E. Hansen

Questions about the program can be addressed to

Bruce E. Hansen
Department of Economics
Social Science Building
University of Wisconsin
Madison, WI 53706-1393
bhansen@ssc.wisc.edu
http://www.ssc.wisc.edu/~bhansen/

*/

/* ******** testing program ******** */
load xx1[389,1] = mon59m.dat ;
load xx2[181,1] = mon46q.dat ;
load xx3[177,1] = mon47q.dat ;
load xx4[533,3] = mon47m.dat ;

@ ----- Create Quarterly M1 and 3M-TB Series from Monthly Series ----- @

xx1q = zeros(4*(rows(xx1)-5)/12,1) ;
xx4q = zeros(rows(xx3),1) ;
jm = 1 ;
jmc = 1 ;
endm = rows(xx1) - 5 ;
do while jm <= endm ;
   xx1q[jmc,.] = meanc(xx1[jm:jm+2,.]) ;
   jmc = jmc + 1 ;
   jm = jm + 3 ;
endo ;
jr = 3 ;
jrc = 1 ;
endr = rows(xx4) - 5 ;
do while jr <= endr ;
   xx4q[jrc,.] = xx4[jr,2] ;
   jrc = jrc + 1 ;
   jr = jr + 3 ;
endo ;

@ ----- Set Sample: 1959.1 - 1990.4 ----- @

ynr = xx3[49:176,1]; @ ynr - Net National Product, Billions of 82$ @
pp = xx2[53:180,1]./xx3[49:176,1] ; @ pp - Implicit Price Deflator, PI@
m1 = xx1q[1:128,1]./pp ; @ m1/p - M1, Billions of $ @
sr = xx4q[49:176,1]  ; @ sr - 3 Month T-Bills @

obs = rows(m1) ;

n = obs ;
md=ln(m1);
ynr=ln(ynr);

/*
"Log of Real Money Balances";y;
"Income and Interest Rates";x;
*/
y=md;
x=ynr~sr;
call main(y,x,4,4,6);
end;


/*************************************************************************
----PROC MAIN
----FORMAT: call  main(y,x,model,choice,k)
----INPUT:      y - depend variable
        x - data matrix for independent variables (first row is
first observation)
                model - choice for model        =2  C
                        =3  C/T
                        =4  C/S
        choice - only in ADF test,  =1  pre-specified AR lag
                        =2  AIC-chosen AR lag
                        =3  BIC-chosen AR lag
                        =4  downward-t-chosen AR lag
        k - maximum lag for ADF test
----OUTPUT: print automatically Za*, breakpoint for Za*, Zt*, breakpoint for Zt*
, ADF*,                        breakpoint for ADF* and AR lag chosen for ADF*
----GLOBAL VARIABLES: none
----EXTERNAL PROCEDURES: adf,  phillips
----NB: Constant included in regression
************************************************************************/

/*
****************  Main procedure *******************
*/

proc(0)=main(y,x,model,choice,k);
   local t,n,final,begin,tstat,x1,lag,j,dummy,temp1,temp2,temp3,temp4;
   local breakpt1,breakpt2,breakpta,za,zt;
   n=rows(y);
   begin=round(0.15*n);
   final=round(0.85*n);
   temp1=zeros(final-begin+1,1);
   temp2=temp1;
   temp3=temp1;
   temp4=temp1;
   t=begin;
   do while t<=final;
     dummy=zeros(t,1)|ones(n-t,1);
     @ adjust regressors for different models @
     if model==3;
        x1=ones(n,1)~dummy~seqa(1,1,n)~x;
     elseif model==4;
        x1=ones(n,1)~dummy~x~dummy.*x;
     elseif model==2;
        x1=ones(n,1)~dummy~x;
     endif;

     @ computer ADF for each t  @
    {temp1[t-begin+1],temp2[t-begin+1]}=adf(y,x1,k,choice);

     @ compute Za or Zt for each t  @
     {temp3[t-begin+1],temp4[t-begin+1]}=phillips(y,x1);
     t=t+1;
   endo;

   @  ADF test @
   tstat=minc(temp1);
   lag=minindc(temp1);
   breakpta=(lag+begin-1)/n;
   lag=temp2[lag];
   print "******** ADF Test ***********";
   print "t-statistic = " tstat;
   print "AR lag = " lag;
   print "break point(ADF) = " breakpta;
   print " ";

   @  Phillips test @
   za=minc(temp3);
   breakpt1=(minindc(temp3)+begin-1)/n;
   zt=minc(temp4);
   breakpt2=(minindc(temp4)+begin-1)/n;
   print "******** Phillips Test ********";
   print "Zt =              " zt;
   print "breakpoint(Zt) =  " breakpt2;
   print "Za =              " za;
   print "breakpoint(Za) =  " breakpt1;
   print " ";
retp;
endp;
@ -------------------------------------------------------------- @


/**********************  PROC ADF  *****************************
**   FORMAT
**          { stat,lag } = adf(y,x)
**   INPUT
**        y - dependent variable
**        x - independent variables
**   OUTPUT
**  stata - ADF statistic
**  lag - the lag length
**   GLOBAL VARIABLES: none
**   EXTERNAL PROCEDURES: estimate
**********************************************************************/

/*
*************** ADF for each breakpoint ********************
*/
proc(2) = adf(y,x,kmax,choice);
   local b,m,e,e1,n,n1,sig2,se,xe,yde,j,tstat,de,temp1,temp2;
   local lag,k,ic,aic,bic;
   @ compute ADF  @
   n=rows(y);
   {b,e,sig2,se}=estimate(y,x);
   de=e[2:n]-e[1:n-1]; @ difference of residuals @

   ic=0;
   k=kmax;
   temp1=zeros(kmax+1,1);
   temp2=zeros(kmax+1,1);
   do while k>=0;
      yde=de[1+k:n-1];
      n1=rows(yde);
      @  set up matrix for independent variable(lagged residuals)  @
      xe=e[k+1:n-1];
      j=1;
      do while j <= k;
         xe=xe~de[k+1-j:n-1-j];
         j=j+1;
      endo;
      {b,e1,sig2,se}=estimate(yde,xe);
      if choice==1;  @ K is pre-specified @
          temp1[k+1]=-1000;   @ set an random negative constant @
          temp2[k+1]=b[1]/se[1];
          break;
      elseif choice==2;  @ K is determined by AIC @
         aic=ln(e1'e1/n1)+2*(k+2)/n1;
         ic=aic;
      elseif choice==3;  @ K is determined by BIC @
         bic=ln(e1'e1/n1)+(k+2)*ln(n1)/n1;
         ic=bic;
      elseif choice==4; @K is determined by downward t @
         if abs(b[k+1]/se[k+1]) >= 1.96 or k==0;
        temp1[k+1]=-1000;    @ set an random negative constant @
            temp2[k+1]=b[1]/se[1];
            break;
    endif;
      endif;
      temp1[k+1]=ic;
      temp2[k+1]=b[1]/se[1];
      k=k-1;
   endo;

   lag=minindc(temp1);
   tstat=temp2[lag];
   retp(tstat,lag-1);
endp;
@ ------------------------------------------------------------ @



/**********************  PROC PHILLIPS  *****************************
**   FORMAT
**  { za,zt } = phillips(y,x)
**   INPUT
**  y  - dependent variable
**  x - independent variables
**   OUTPUT
**  za - the Phillips test statistic
**  zt -  the Phillips test statistic
**   GLOBAL VARIABLES: none
**********************************************************************/

/*
*************** Za or Zt for each breakpoint ********************
*/
proc(2)=phillips(y,x);
   local n,b,e,be,ue,nu,bu,uu,su,a2,bandwidth,m,j;
   local c,lemda,gama,w,p,sigma2,s,za,zt;
   n=rows(y);

   @  OLS regression  @
   b=y/x;
   e=y-x*b;

   @  OLS regression on residuals @
   be=e[2:n]/e[1:n-1];
   ue=e[2:n]-e[1:n-1]*be;

   @ calculate bandwidth number @
   nu=rows(ue);
   bu=ue[2:nu]/ue[1:nu-1];
   uu=ue[2:nu]-ue[1:nu-1]*bu;
   su=meanc(uu.^2);
   a2=(4*bu^2*su/(1-bu)^8)/(su/(1-bu)^4);
   bandwidth=1.3221*((a2*nu)^0.2);

   m=bandwidth;
   j=1;
   lemda=0;
   do while j<=m;
      gama=ue[1:nu-j]'ue[j+1:nu]/nu;
      c=j/m;
      w=(75/(6*pi*c)^2)*(sin(1.2*pi*c)/(1.2*pi*c)-cos(1.2*pi*c));
      lemda=lemda+w*gama;
      j=j+1;
   endo;

   @ calculate Za and Zt for each t @
   p=sumc(e[1:n-1].*e[2:n]-lemda)/sumc(e[1:n-1].^2);
   za=n*(p-1);
   sigma2=2*lemda+ue'ue/nu;
   s=sigma2/(e[1:n-1]'e[1:n-1]);
   zt=(p-1)/sqrt(s);
   retp(za,zt);
endp;
@ ------------------------------------------------------------ @


/**********************  PROC ESTIMATE  *****************************
**   FORMAT
**          { b,e,sig2,se } = estimate(y,x)
**   INPUT
**        y  - dependent variable
**        x - independent variables
**   OUTPUT
**  b - OLS estimates
**  e - residuals
**  sig2 - variance
**  se - standard error for coefficients
**   GLOBAL VARIABLES: none
**********************************************************************/
/* *****  ols regression ****** */
proc(4) = estimate(y,x);
   local m, b, e, sig2, se;
   m=invpd(moment(x,0));
   b=m*(x'y);
   e=y-x*b;
   sig2=(e'e)/(rows(y)-cols(x));
   se=sqrt(diag(m)*sig2);
   retp(b,e,sig2,se);
endp;
@ ---------------------------------------------------------------- @

