%********************************************************
%UR.M

%written by
%Bruce E. Hansen
%Department of Economics
%University of Wisconsin
%www.ssc.wisc.edu/~bhansen

%This program produces the Fan Chart (Figure 2) from
%"Interval Forecasts and Parameter Uncertainty"
%August, 2004

%*********************************************************/
function fan()
load lhc.txt;
load lhuem.txt;
dat=lhuem./lhc*100;
n=length(dat(:,1));
hh=[1 3 6 9 12]';
hn=length(hh);
ss=[60 120 180 240]';
sn=length(ss);
kmax=12;



alpha=[.1 .9]';
store1=zeros(hn,sn);
store2=store1;
store3=store1;
store4=store1;

for hi=1:hn
    for si=1:sn 
        sample=ss(si);
        horizon=hh(hi);
        forecast1=zeros(n-sample-horizon+1,1);
        forecast2=zeros(n-sample-horizon+1,1);
        forecast3=zeros(n-sample-horizon+1,1);
        forecast4=zeros(n-sample-horizon+1,1);
        yf=dat(sample+horizon:n);
        for t=1:n-sample-horizon+1
            yt=yf(t);
            datt=dat(t:t+sample-1);
            k=aic(datt,horizon,kmax);
            y=datt(horizon+k:sample);
            x=datt(k:sample-horizon);
            for i=2:k
                x=[x,datt(k-i+1:sample-horizon-i+1)];
            end;
            ny=length(y(:,1));
            xn=y((ny:-1:ny-k+1));
            [q1,q2,q3,q4]=qinterval(y,x,xn,horizon,alpha);
            forecast1(t)=(yt<=q1(2)).*(yt>=q1(1));
            forecast2(t)=(yt<=q2(2)).*(yt>=q2(1));
            forecast3(t)=(yt<=q3(2)).*(yt>=q3(1));
            forecast4(t)=(yt<=q4(2)).*(yt>=q4(1));
        end;
        store1(hi,si)=mean(forecast1)';
        store2(hi,si)=mean(forecast2)';
        store3(hi,si)=mean(forecast3)';
        store4(hi,si)=mean(forecast4)';
    end;
end;
out=fopen('urout.txt','wt');
ss=[0,ss'];
fprintf(out,'Rough Interval (Nominal .80)\n\n');
pss=[ss;[hh,store1]];
for i=1:length(pss(:,1))
    for j=1:length(pss(1,:))
        fprintf(out,'%f     ',pss(i,j));
    end;
    fprintf(out,'\n');
end;

fprintf(out,'Simple Reference Interval (Nominal .80)\n\n');
pss=[ss;[hh,store2]];
for i=1:length(pss(:,1))
    for j=1:length(pss(1,:))
        fprintf(out,'%f     ',pss(i,j));
    end;
    fprintf(out,'\n');
end;

fprintf(out,'Convolution Interval (Nominal .80)\n\n');
pss=[ss;[hh,store3]];
for i=1:length(pss(:,1))
    for j=1:length(pss(1,:))
        fprintf(out,'%f     ',pss(i,j));
    end;
    fprintf(out,'\n');
end;

fprintf(out,'NonParametric Interval (Nominal .80)\n\n');
pss=[ss;[hh,store4]];
for i=1:length(pss(:,1))
    for j=1:length(pss(1,:))
        fprintf(out,'%f     ',pss(i,j));
    end;
    fprintf(out,'\n');
end;
fclose(out);        

% Plot forecast intervals over time against actual %
sample=180;
horizon=3;

years=(1948+(sample+horizon-1)/12:1/12:(n-sample-horizon)/12+1948+(sample+horizon-1)/12);
forecasts=zeros(n-sample-horizon+1,3)+NaN;
forecasts(:,1)=dat(sample+horizon:n);
for t=1:n-sample-horizon+1
    datt=dat(t:t+sample-1);
    k=aic(datt,horizon,kmax);
    y=datt(horizon+k:sample);
    x=datt(k:sample-horizon);
    for i=2:k
        x=[x,datt(k-i+1:sample-horizon-i+1)];
    end;
    ny=length(y(:,1));
    xn=y((ny:-1:ny-k+1));
    [q1,q2,q3,q4]=qinterval(y,x,xn,horizon,alpha);
    forecasts(t,2:3)=q2';
end;
plot(years,forecasts(:,3),'b-',years,forecasts(:,1),'k--',years,forecasts(:,2),'g-.');
title('80% Forecast Intervals');
xlabel(strcat('Sample Size = ',num2str(sample),',  Forecast Horizon = ',num2str(horizon)));
ylabel('Unemployment Rate');  
xlim([1970 2005]);
ylim([3 12]);
legend('90% Forecast Quantile','Actual','Forecast Quantile');
saveas(gcf,'ur.eps','psc2');


%****AIC****************************
%Computes lag order using AIC 
%*****************************************/
function k=aic(dat,horizon,kmax);
n=length(dat);
t=n-horizon-kmax+1;
y=dat(horizon+kmax:n);
ab=zeros(kmax,1);
x=ones(t,1);
for i=1:kmax
    x=[x,dat(kmax-i+1:n-horizon-i+1)];
    e=y-x*(y'/x')';
    ab(i)=log(e'*e/t)+2*(1+i)/t;
end;
[temp,k]=min(ab);


%*******************************************************
%Function qinterval

%Computes interval endpoints (quantiles) with correction for
%sampling uncertainty.

%The forecast model takes the form

%y = a + x*b + e

%We are forecasting y h-steps ahead, given x=xn

%Format: {q1,q2,q3,q4}=Qinterval(y,x,xn,horizon,alpha);

%Inputs: y	nx1 	forecast variable
%	x	nxk 	regressors (no constant)
%	xn	kx1 	regressors for forecast
%	horizon	1x1 	forecast horixon
%			This is used only to determine bandwidth for variance estimation
%	alpha	px1	list of quantiles, endpoints for forecast interval

%Output:	q1	px1 	rough forecast quantiles
%	q2	px1	simple reference adjustment forecast quantiles
%	q3	px1	convolution adjustment forecast quantiles
%	q4	px1	nonparametric adjustment forecast quantiles


%*****************************************/

function [q1,q2,q3,q4]=qinterval(y,x,xn,horizon,alpha);
n=length(y(:,1));
x1=[ones(n,1),x];
m1=x1'*x1;
xn1=[1;xn];
xn1=xn1-mean(x1)';
beta=(m1)^(-1)*(x1'*y);
e=y-x1*beta;
  
% Empirical quantiles of residuals @
  
ee=sortrows(e,1);
bc=n*alpha;
i=floor(bc);
i=i+(i == 0)-(i == n);
q=ee(i,:).*(i == bc) + ((ee(i,:)+ee(i+1,:))/2).*(i ~= bc);
% Point Forecast @
xf=[1;xn]'*beta;

% Density Estimation @
sd=std(e)';
qe=ones(length(e),1)*q'-e*ones(1,length(q));
s0=1.06*sd/(n^(.2));
s2=.94*sd/(n^(1/9));
s3=.93*sd/(n^(1/11));
f0=mean(normpdf(qe/s0,0,1))'/s0;
q2=qe/s2;
f2=mean(normpdf(q2,0,1).*(q2.^2-1))'/(s2^2);
q3=qe/s3;
f3=mean(normpdf(q3,0,1).*(3*q3-q3.^3))'/(s3^2);
r0=((f0./(2*sqrt(pi)*n*(f2.^2))).^(.2));
r1=((.75*f0./(sqrt(pi)*n*(f3.^2))).^(1/7));
f=mean(normpdf(qe./(ones(length(qe(:,1)),1)*r0'),0,1))'./r0;
f1=-mean(normpdf(qe./(ones(length(qe(:,1)),1)*r1'),0,1).*qe)'./(r1.^3);

% Variance Estimation @
u=((e*ones(1,length(q))<=ones(length(e),1)*q')-ones(length(e),1)*alpha')./(ones(length(e),1)*f')-(x1.*(e*ones(1,length(x1(1,:)))))*((m1/n)^(-1)*xn1)*ones(1,length(q));
na=length(alpha(:,1));
s=zeros(na,1);
for i=1:na
    gi=u(:,i);
    si = gi'*gi;
    v=si*.01/n/n;
    for j=1:(horizon-1)
         si = si + 2*(gi(1:n-j)'*gi(1+j:n));
    end;
    si=si/n/n;
    if si<=v; si=0; end;
    s(i)=si;
end;

% Forecast Quantiles @
q1=xf+q;
q2=q1+(s.*q)*n/2/(e'*e);
q3=xf+SmoothQuantile(e,alpha,sqrt(s));
q4=q1-s.*f1./f/2;

%*****************************************
%PROC SmoothQUANTILE

%Format: q=SmoothQuantile(x,a,s);
%Inputs: x	nx1 data 
%	a	px1 list of quantiles, e.g. a=.2
%	s	px1 list of standard deviations
%Output:	q	px1 matrix of quantiles

%This procedure estimates quantiles from a univariate distribution from a 
%the empirical distribution function smoothed with a normal kernel with 
%given standard deviations (bandwidths).
%
%If s=0, the procedure returns the unsmoothed empirical quantile.

%*****************************************/
function qq=SmoothQuantile(x,alpha,s);
xx=sortrows(x,1);
bc=length(x(:,1))*alpha;
i=floor(bc);
i=i+(i == 0)-(i == length(x(:,1)));
q=xx(i,:).*(i == bc) + ((xx(i,:)+xx(i+1,:))/2).*(i ~= bc);
na=length(alpha(:,1));
qq=q;
for i=1:na
    a=alpha(i);
    qi=q(i);
    if length(s(:,1))==na; si=s(i); elseif length(s(:,1))==1; si=s; end;    
    for j=1:20;
        if si>0;
            qs=(qi-x)./si;
            f=mean(normcdf(qs,0,1)-a)';
            af=abs(f);
            if af < .0001;
                qq(i)=qi;break;
            else;
                d=f*si/mean(normpdf(qs,0,1))';
                ind=0;
                while ind==0
                    qt=qi-d;
                    aft=abs(mean(normcdf((qt-x)./si,0,1)-a)');
                    if aft<af
                        qi=qt;
                        ind=1;
                    else
                        d=d/2;
                    end;
                %retry:      
                %qt=qi-d;
                %aft=abs(meanc(cdfn((qt-x)./si)-a));
                %if aft<af;
                %    qi=qt;
                %else;
                %    d=d/2;
                %    goto retry;
                %end;
                end;
            end;
        end;
    end;    
end;