/********************************************************
QINTERVAL.PRC

written by
Bruce E. Hansen
Department of Economics
University of Wisconsin
www.ssc.wisc.edu/~bhansen

This program implements the methods reported in
"Interval Forecasts and Parameter Uncertainty"
Augustr, 2004

*********************************************************/


/*******************************************************
PROC qinterval

Computes interval endpoints (quantiles) with correction for
sampling uncertainty.

The forecast model takes the form

y = a + x*b + e

We are forecasting y h-steps ahead, given x=xn

Format: {q1,q2,q3,q4}=Qinterval(y,x,xn,horizon,alpha);

Inputs: y	nx1 	forecast variable
	x	nxk 	regressors (no constant)
	xn	kx1 	regressors for forecast
	horizon	1x1 	forecast horixon
			This is used only to determine bandwidth for variance estimation
	alpha	px1	list of quantiles, endpoints for forecast interval

Output:	q1	px1 	rough forecast quantiles
	q2	px1	simple reference adjustment forecast quantiles
	q3	px1	convolution adjustment forecast quantiles
	q4	px1	nonparametric adjustment forecast quantiles


*****************************************/

proc (4) = qinterval(y,x,xn,horizon,alpha);
local n,x1,xn1,m1,beta,e,ee,bc,i,q,xf,h,f,f1,x2,u,na,s,gi,si,v,j,q1,q2,q3,q4,sd,
s0,s2,s3,f0,f2,f3,r0,r1,qe;
  n=rows(y);
  x1=ones(n,1)~x;
  m1=moment(x1,0);
  xn1=1|xn;
  xn1=xn1-meanc(x1);
  beta=solpd(x1'y,m1);
  e=y-x1*beta;
  @ Empirical quantiles of residuals @
  ee=sortc(e,1);
  bc=n*alpha;
  i=floor(bc);
  i=i+(i .== 0)-(i .== n);
  q=ee[i,.].*(i .== bc) + ((ee[i,.]+ee[i+1,.])/2).*(i .ne bc);
  @ Point Forecast @
  xf=(1|xn)'beta;

  @ Density Estimation @
  sd=stdc(e);
  qe=q'-e;
  s0=1.06*sd/(n^(.2));
  s2=.94*sd/(n^(1/9));
  s3=.93*sd/(n^(1/11));
  f0=meanc(pdfn(qe/s0))/s0;
  q2=qe/s2;
  f2=meanc(pdfn(q2).*(q2.^2-1))/(s2^2);
  q3=qe/s3;
  f3=meanc(pdfn(q3).*(3*q3-q3.^3))/(s3^2);
  r0=((f0./(2*sqrt(pi)*n*(f2.^2))).^(.2));
  r1=((.75*f0./(sqrt(pi)*n*(f3.^2))).^(1/7));
  f=meanc(pdfn(qe./r0'))./r0;
  f1=-meanc(pdfn(qe./r1').*qe)./(r1^3);

  @ Variance Estimation @
  u=((e.<=q')-alpha')./(f')-(x1.*e)*solpd(xn1,m1/n);
  na=rows(alpha);
  s=zeros(na,1);
  for i (1,na,1);
    gi=u[.,i];
    si = gi'gi;
    v=si*.01/n/n;
    for j (1,horizon-1,1);
      si = si + 2*(gi[1:n-j]'gi[1+j:n]);
    endfor;
    si=si/n/n;
    if si<=v; si=0; endif;
     s[i]=si;
  endfor;

  @ Forecast Quantiles @
  q1=xf+q;
  q2=q1+(s.*q)*n/2/(e'e);
  q3=xf+SmoothQuantile(e,alpha,sqrt(s));
  q4=q1-s.*f1./f/2;
retp(q1,q2,q3,q4);
endp;

/*****************************************
PROC SmoothQUANTILE

Format: q=SmoothQuantile(x,a,s);
Inputs: x	nx1 data 
	a	px1 list of quantiles, e.g. a=.2
	s	px1 list of standard deviations
Output:	q	px1 matrix of quantiles

This procedure estimates quantiles from a univariate distribution from a 
the empirical distribution function smoothed with a normal kernel with 
given standard deviations (bandwidths).

If s=0, the procedure returns the unsmoothed empirical quantile.

*****************************************/
proc SmoothQuantile(x,alpha,s);
local xx,bc,i,q,na,qq,a,qi,f,si,qs,j,af,d,qt,aft;
  xx=sortc(x,1);
  bc=rows(x)*alpha;
  i=floor(bc);
  i=i+(i .== 0)-(i .== rows(x));
  q=xx[i,.].*(i .== bc) + ((xx[i,.]+xx[i+1,.])/2).*(i .ne bc);
  na=rows(alpha);
  qq=q;
  for i (1,na,1);
    a=alpha[i];
    qi=q[i];
    if rows(s)==na; si=s[i]; elseif rows(s)==1; si=s; endif;    
    for j (1,20,1);
     if si>0;
      qs=(qi-x)./si;
      f=meanc(cdfn(qs)-a);
      af=abs(f);
      if af < .0001;
        qq[i]=qi;break;
      else;
        d=f*si/meanc(pdfn(qs));
        retry:      
        qt=qi-d;
        aft=abs(meanc(cdfn((qt-x)./si)-a));
        if aft<af;
          qi=qt;
        else;
          d=d/2;
          goto retry;
        endif;
      endif;
     endif;
    endfor;    
  endfor;
retp(qq);
endp;



