#########################################################################
##  BREAK.R
##  
##  This file contains the R procedure BREAK_T written by
##  
##  Bruce E. Hansen
##  Department of Economics
##  Social Science Building
##  University of Wisconsin
##  Madison, WI 53706-1393
##  behansen@wisc.edu
##  http://www.ssc.wisc.edu/~bhansen/
##  
##  The procedure implements the testing methods dicussed in
##  "Testing for Structural Change in Conditional Models."
##  
##  The file calls the procedures PV_SUP, PV_EXP and PV_AVE,
##  which are contained in separate ASCII files
##  
##  Format:   out <- break_t(y,x,qvar,t1,t2)
##  
##  There are no returns from the procedure. Results are printed to the screen.
##  
##  y    = dependent variable (nx1 vector)
##  x    = regressors (nxk matrix, may contain lagged y's)
##  qvar = (k+1)x1 string vector, containing names of y and x variables
##         First element contains name of y, remainder names of x's
##  t1   = starting breakpoint index or percentage,
##         may be integer in [k+1,T-k-1] or percentage in (0,1).
##  t2   = ending breakpoint index or percentage,
##         may be integer in [k+1,T-k-1] or percentage in (0,1),
##         must equal or exceed t1.
##         Note: For Sup test, Andrews recommends t1=.15 and t2=.85
##               For Exp test, Andrews-Ploberger recommend t1=.02 and t2=.98
##  
#########################################################################

source ("pv_sup.R")
source ("pv_exp.R")
source ("pv_ave.R")

break_t <- function(y,x,qvar,t1,t2){

r1 <- 10
r2 <- 100
# Note: Number of Bootstrap Replications is r1*r2.
#       The program works in blocks of r2 at a time.
#       The higher is r2, the faster the program will be,
#       but the more memory is required.  

k <- ncol(x)
n <- nrow(y)

if (n != nrow(x)) {
    cat ("ERROR:  Number of nrow in Y and X must equal.","\n")
    cat ("        Please re-specify.","\n")
}
if (t1<1){  
    n1 <- floor(n*t1)  
    tau1 <- t1
}else{  
    n1 <- t1  
    tau1 <- t1/n
}
if (t2<1){  
    n2 <- floor(n*t2) 
    tau2 <- t2
}else{  
    n2 <- t2  
    tau2 <- t2/n
}

cat ("Dependent Variable              = ", qvar[1], "\n")
cat ("\n")
cat ("Starting Index for Break Search = ", n1, "\n")
cat ("   (Percentage)                 = ", tau1, "\n")
cat ("Ending Index for Break Search   = ", n2, "\n")
cat ("   (Percentage)                 = ", tau2, "\n")
if (n1 <= k){
  cat ("ERROR:  Starting Sample is smaller than Number of Parameters","\n")
  cat ("        You need to select a larger value for t1","\n")
}
if (n2 >= (n-k)){
  cat ("ERROR:  Ending Sample is smaller than Number of Paramters","\n")
  cat ("        You need to select a smaller value for t2","\n")
}

  xx <- t(x)%*%x 
  xxi <- solve(xx)
  beta <- xxi%*%(t(x)%*%y)
  e <- y - x%*%beta
  xe <- x*(e%*%matrix(1,1,ncol(x)))
  ee <- t(e)%*%e
  sig <- ee/(n-k)
  se <- as.matrix(sqrt(diag(xxi)*sig))
  yd <- y-mean(y)
  r_2 <- 1 - ee/(t(yd)%*%yd)
  cat ("\n")
  cat ("Full Sample Estimation", "\n")
  cat ("Sample Size                     = ", n, "\n")
  cat ("Number of Regressors            = ", k, "\n")
  cat ("Sample Variance                 = ", sig, "\n")
  cat ("R-squared                       = ", r_2, "\n")
  cat ("\n")
  cat ("\n")
  cat ("Estimates and Standard Errors from Full Sample: ", "\n")
  cat ("\n")
  Tqvar <- format(qvar,digits=4)
  Tbeta <- format(beta,digits=4)
  Tse <- format(se,digits=4)
  for (j in 1:k) cat (Tqvar[j+1]," ",Tbeta[j]," ",Tse[j],"\n")
  cat ("\n")
  cat ("\n")
  cat ("\n")
  f  <- matrix(0,n,1)
  m <- t(x[1:(n1-1),])%*%(x[1:(n1-1),])
  mi <- solve(m)  
  msi <- solve(xx-m) 
  sn <- as.matrix(colSums(xe[1:(n1-1),]))
    for (ib in n1:n2){ 
      xi <- as.matrix(x[ib,])
      xim <- t(xi)%*%mi
      mi <- mi - (t(xim)%*%xim)/as.vector(1+xim%*%xi)
      xim <- t(xi)%*%msi
      msi <- msi + (t(xim)%*%xim)/as.vector(1-xim%*%xi)
      sn <- sn + as.matrix(xe[ib,])
      q <- t(sn)%*%msi%*%xx%*%mi%*%sn
      f[ib] <- q%*%(n-k*2)/(ee-q)
    }
    im <- which.max(f)
    supf <- f[im]
    avef <- mean(f[n1:n2])
    expf <- log(mean(exp(f[n1:n2]/2)))

  # Standard P-Value #
  l_0 <- tau2*(1-tau1)/(tau1*(1-tau2))
  pi_0 <- 1/(1+sqrt(l_0))
  pv_s <- pv_sup(supf,k,l_0)
  pv_e <- pv_exp(expf,k,l_0)
  pv_a <- pv_ave(avef,k,l_0)

  x1 <- x[1:im,]
  x2 <- x[(im+1):n,]
  y1 <- as.matrix(y[1:im])
  y2 <- as.matrix(y[(im+1):n])
  xx1 <- solve(t(x1)%*%x1)  
  xx2 <- solve(t(x2)%*%x2)  
  beta1 <- xx1%*%(t(x1)%*%y1)
  beta2 <- xx2%*%(t(x2)%*%y2)
  e1 <- y1-x1%*%beta1
  e2 <- y2-x2%*%beta2
  sig1 <- (t(e1)%*%e1)/(im-k)
  sig2 <- (t(e2)%*%e2)/(n-im-k)
  se1  <- as.matrix(sqrt(diag(xx1)*sig1))
  se2  <- as.matrix(sqrt(diag(xx2)*sig2))
  em <- rbind(e1,e2)
  yd1 <- y1-mean(y1)
  yd2 <- y2-mean(y2)
  r21 <- 1 - (t(e1)%*%e1)/(t(yd1)%*%yd1)
  r22 <- 1 - (t(e2)%*%e2)/(t(yd2)%*%yd2)
  cat ("Estimated Breakpoint (index)          = ", im, "\n")
  cat ("Percentage of Sample                  = ", (im/n), "\n")
  cat ("\n")
  cat ("First Sample:" ,"\n")
  cat ("Number of Observations                = ", im, "\n")
  cat ("Sample Variance                       = ", sig1, "\n")
  cat ("R-squared                             = ", r21, "\n")
  cat ("\n")
  cat ("\n")
  cat ("Estimates and Standard Errors from First Sample: ", "\n")
  cat ("\n")
  Tbeta1 <- format(beta1,digits=4)
  Tse1 <- format(se1,digits=4)
  for (j in 1:k) cat (Tqvar[j+1]," ",Tbeta1[j]," ",Tse1[j],"\n")
  cat ("\n")
  cat ("\n")
  cat ("Second Sample:","\n")
  cat ("Number of Observations                = ", (n-im),"\n")
  cat ("Sample Variance                       = ", sig2,"\n")
  cat ("R-squared                             = ", r22,"\n")
  cat ("\n")
  cat ("\n")
  cat ("Estimates and Standard Errors from Second Sample: ","\n")
  cat ("\n")
  Tbeta2 <- format(beta2,digits=4)
  Tse2 <- format(se2,digits=4)
  for (j in 1:k) cat (Tqvar[j+1]," ",Tbeta2[j]," ",Tse2[j],"\n")
  cat ("\n")
  cat ("\n")


# Fixed Regressor Bootstrap #

supfb <- matrix(0,r2,r1)   
expfb <- matrix(0,r2,r1) 
avefb <- matrix(0,r2,r1) 
supfh <- matrix(0,r2,r1) 
expfh <- matrix(0,r2,r1) 
avefh <- matrix(0,r2,r1) 

for (ri in 1:r1){

    u   <- matrix(rnorm(n*r2),n,r2)  
    euf <- u - x%*%xxi%*%(t(x)%*%u)
    eef <- as.matrix(colSums(euf^2))
    ff  <- matrix(0,n,r2)  
    snf <- t(x[1:(n1-1),])%*%euf[1:(n1-1),]

    uh  <- u*(em%*%matrix(1,1,ncol(u)))
    euh <- uh - x%*%xxi%*%(t(x)%*%uh)
    eeh <- as.matrix(colSums(euh^2))
    ffh <- matrix(0,n,r2) 
    snh <- t(x[1:(n1-1),])%*%euh[1:(n1-1),]

    mf <-  t(x[1:(n1-1),])%*%x[1:(n1-1),]
    mif <- solve(mf)
    msif <- solve(xx-mf)

    for (ib in n1:n2){

      xif <- as.matrix(x[ib,])
      ximf <- t(xif)%*%mif
      mif <- mif - (t(ximf)%*%ximf)/as.vector(1+ximf%*%xif)
      ximf <- t(xif)%*%msif
      msif <- msif + (t(ximf)%*%ximf)/as.vector(1-ximf%*%xif)    
      snf  <- snf + xif%*%t(as.matrix(euf[ib,]))
      qf   <- as.matrix(colSums(snf*(msif%*%xx%*%mif%*%snf)))
      ff[ib,] <- t(qf/(eef-qf)) 
      snh  <- snh + xif%*%t(as.matrix(euh[ib,]))
      qf   <- as.matrix(colSums(snh*(msif%*%xx%*%mif%*%snh)))
      ffh[ib,] <-t(qf/(eeh-qf))
    }

    ftf <- ff[n1:n2,]*(n-k*2)
    supfb[,ri] <- (apply(ftf,2,max) > supf)
    expfb[,ri] <- (colMeans(exp(ftf/2)) > exp(expf))
    avefb[,ri] <- (colMeans(ftf) > avef)

    ftf <- ffh[n1:n2,]*(n-k*2)
    supfh[,ri] <- (apply(ftf,2,max) > supf)
    expfh[,ri] <- (colMeans(exp(ftf/2)) > exp(expf))
    avefh[,ri] <- (colMeans(ftf) > avef)

}

supfb <- mean(supfb)
expfb <- mean(expfb)
avefb <- mean(avefb)

supfh <- mean(supfh)
expfh <- mean(expfh)
avefh <- mean(avefh)

cat ("Tests for Structural Change ","\n")
cat ("\n")
cat ("Bootstrap Replications  = ", r1*r2,"\n")
cat ("\n")
cat ("        ","Test        ","Andrews      ","Bootstrap    ","Hetero-Corrected","\n")
cat ("        ","Statistic   ","P-Value      ","P-Value      ","P-Value         ","\n")
cat ("\n")
cat ("\n")
cat ("SupF    ", supf,"   ",pv_s,"   ",supfb,"   ",supfh,"\n")
cat ("ExpF    ", expf,"   ",pv_e,"   ",expfb,"   ",expfh,"\n")
cat ("AveF    ", avef,"   ",pv_a,"   ",avefb,"   ",avefh,"\n")
cat ("\n")
cat ("\n")
}


