%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This file contains two MATLAB procedures:
% PV_AVE and AVE_BETA

%    written by

% Bruce E. Hansen
% Department of Economics
% Social Science Building
% University of Wisconsin
% Madison, WI 53706-1393
% bhansen@ssc.wisc.edu
% http://www.ssc.wisc.edu/~bhansen/
    

% The format for the procedures are

% p = pv_ave(Tn,m,l);
% b = ave_beta(k);

% The documentation follows.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% PV_AVE

% Procedure to compute asymptotic p-values for Andrews-Ploberger
% Ave Test based on approximation used in my paper
% "Approximate Asymptotic P-Values for Structural Change Tests".

% Format:

% p = pv_ave(Tn,m,l);

% Inputs:
% Tn   =  Value of Ave Statistic
% m    =  Number of parameters tested for constancy
%         (degrees of freedom of conventional Chow test)
% l    =  Either the "lambda" or the "pi_0" of Andrews-Ploberger
%     Lambda lies between [1,infinity) and pi_0 between (0,1/2].
%     pi_0 is convenient when the test is constructed using symmetric
%     trimming, and Lambda is convenient when non-symmetric trimming is
%     used.

% Output:
% p    =  Asymptotic p-value of test statistic



% EXP_BETA

% Procedure to return coefficients for computation of p-value function
% for the Exp test, given degrees of freedom.

% Format:

% beta = exp_beta(k);

% Input:
% k  = degrees of freedom (number of parameters tested for constancy)

% Output:
% beta = 25xg matrix.
%        Each row consists of coefficients for p-value function.
%        The rows correspond to pi_0 indexed from .49 to .01 (decreasing)
%        in steps of .02.  The number of columns, g, depends on k.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function p=pv_ave(tn,k,l);
if l<1
    tau=l;
else
    tau=1/(1+sqrt(l));
end;
if k==1
    m=4;
elseif k==2
    m=3;
else
    m=2;
end;

beta=ave_beta(k);
x=beta(:,1:m)*(tn.^(0:m-1)');
x=x.*(x>0);
pp=1-chi2cdf(x,beta(:,m+1));

if tau==.5
    p=1-chi2cdf(tn,k);
elseif tau<=.01
    p=pp(25);
elseif tau>=.49
    p=((.5-tau)*pp(1)+(tau-.49)*(1-chi2cdf(tn,k)))*100;
else
    taua=(.5-tau+.01)*50;
    tau1=floor(taua);
    p=(tau1+1-taua)*pp(tau1)+(taua-tau1)*pp(tau1+1);
end;

function beta=ave_beta(k);
if k == 1;
 beta=[
   -0.01837180    0.98624967    0.00540622   -0.00073801    0.97463664
   -0.05119386    1.05623887   -0.01352573    0.00188900    0.98812242
   -0.07867301    1.12745039   -0.02774934    0.00397645    1.01197499
   -0.10590321    1.23321648   -0.04927701    0.00631328    1.06011812
   -0.13361664    1.35573164   -0.07343441    0.00868387    1.11824667
   -0.16194620    1.49312113   -0.10195186    0.01136081    1.18485003
   -0.19108078    1.64595807   -0.13548426    0.01478916    1.26021895
   -0.21789738    1.79369437   -0.16473557    0.01755301    1.33946687
   -0.24679207    1.94101915   -0.18960486    0.01945320    1.41971458
   -0.26774640    2.05886200   -0.20121180    0.01949951    1.49384726
   -0.29296559    2.25496797   -0.24521005    0.02412351    1.60546300
   -0.32519388    2.42619868   -0.28825633    0.02965918    1.69572294
   -0.34748020    2.55613596   -0.30209625    0.02974910    1.77985998
   -0.35602069    2.79138446   -0.34251938    0.03217571    1.94323515
   -0.37388507    2.99035388   -0.37685141    0.03417233    2.07433765
   -0.40098411    3.23551216   -0.42813033    0.03884319    2.22607538
   -0.42827889    3.51097689   -0.49786905    0.04658620    2.39037059
   -0.46752042    3.62959081   -0.50981974    0.04675369    2.45781286
   -0.51401517    3.82155126   -0.55877563    0.05417598    2.55803741
   -0.56814452    3.84300346   -0.53706136    0.05077601    2.55903266
   -0.61051723    4.13483986   -0.60773228    0.05895169    2.73127436
   -0.66465293    4.54369777   -0.72403383    0.07412163    2.96102620
   -0.74248417    4.94897981   -0.84287945    0.09009093    3.16541195
   -0.85453578    5.11509052   -0.86825039    0.09312986    3.20551379
   -1.01814056    5.39001882   -0.95098772    0.10541077    3.24921192];
elseif k == 2;
 beta=[
   -0.03220343    0.95140320    0.00628734    1.92151377
   -0.06990945    0.99681586    0.00514929    1.96307128
   -0.12149347    1.03730419    0.00423562    1.98616415
   -0.18758275    1.04824072    0.00575331    1.95362725
   -0.25864848    1.04797528    0.00803372    1.90059873
   -0.31513679    1.06910128    0.00867592    1.89305029
   -0.36420744    1.12469785    0.00687567    1.94475373
   -0.40800873    1.21428279    0.00204996    2.04876539
   -0.44875061    1.30904146   -0.00329275    2.16486989
   -0.48669972    1.37573668   -0.00536157    2.24765179
   -0.53034987    1.40136325   -0.00284433    2.26938907
   -0.57432345    1.57531020   -0.01542791    2.49497492
   -0.61827483    1.70414268   -0.02450148    2.65714696
   -0.68883555    1.80386565   -0.03107986    2.75346410
   -0.73570307    1.92570499   -0.03838190    2.90893677
   -0.79277911    2.03833930   -0.04380117    3.04663595
   -0.86917412    2.12475644   -0.04791593    3.12426054
   -0.93674836    2.20823310   -0.05079253    3.21266306
   -1.03250868    2.29218921   -0.05438482    3.27348114
   -1.12149975    2.36487555   -0.05634662    3.32613419
   -1.21405656    2.50138137   -0.06375229    3.47050282
   -1.30856480    2.64567828   -0.07128089    3.62825212
   -1.42295544    2.79853731   -0.07998139    3.77972930
   -1.58384946    2.93185241   -0.08804021    3.85319929
   -1.77752732    3.12279311   -0.10031811    3.98601933];
elseif k == 3;
 beta=[
    0.01425507    1.01988254    3.05401647
   -0.10010962    1.03922071    3.00266828
   -0.20116025    1.06280563    2.97483237
   -0.29619658    1.08362069    2.94443497
   -0.37224016    1.11319301    2.95847089
   -0.46041270    1.14180473    2.95816958
   -0.52153148    1.18285406    3.01914274
   -0.59086632    1.21805910    3.05701854
   -0.66409757    1.26226915    3.11631679
   -0.74458539    1.29586408    3.14091462
   -0.83954205    1.32613699    3.14144197
   -0.92220550    1.36598375    3.18045044
   -1.00707535    1.40329341    3.20979247
   -1.11370982    1.44517290    3.23239977
   -1.19410468    1.48790085    3.28017807
   -1.26937633    1.55886325    3.41441080
   -1.33940353    1.63638681    3.57282530
   -1.41375998    1.70264308    3.69647183
   -1.50857556    1.77267016    3.81117799
   -1.61308851    1.84414350    3.92040211
   -1.73664791    1.91281934    4.00465323
   -1.87576057    1.98230324    4.07600499
   -2.02695590    2.06191316    4.16653533
   -2.20488362    2.14247295    4.23264999
   -2.40811389    2.24215215    4.32757572];
elseif k == 4;
 beta=[
   -0.02448730    1.02800413    4.07244304
   -0.17926044    1.04474773    3.98853277
   -0.27098814    1.07428817    4.01496730
   -0.38435455    1.10191642    4.01429619
   -0.48490798    1.12643950    4.01346339
   -0.53428650    1.18194003    4.17515171
   -0.67676535    1.19865798    4.10677746
   -0.72029048    1.25693314    4.28839230
   -0.83647439    1.29437338    4.32431829
   -0.94790229    1.32826683    4.34939808
   -1.04263327    1.37045049    4.42476108
   -1.19048104    1.40685020    4.42666778
   -1.28963132    1.45076160    4.50355452
   -1.41328970    1.49705250    4.56677365
   -1.52061945    1.54846081    4.66638631
   -1.68043909    1.58626933    4.66223760
   -1.81397008    1.63061396    4.70837959
   -1.93508573    1.68351074    4.79960209
   -2.07124895    1.74580812    4.91317111
   -2.20454097    1.81362169    5.04955575
   -2.37101474    1.87773308    5.14096621
   -2.55658806    1.94465883    5.22546299
   -2.75850942    2.02224260    5.33477389
   -3.00839705    2.10581631    5.42058343
   -3.27518940    2.19698186    5.51980774];
elseif k == 5;
 beta=[
    0.02083331    1.02517810    5.13305729
   -0.19544528    1.03707100    4.98626304
   -0.31748098    1.06774992    5.01636162
   -0.45111270    1.09349173    5.01314444
   -0.50819790    1.14221548    5.19515594
   -0.64844386    1.16959323    5.19732120
   -0.78388983    1.19830480    5.20923716
   -0.89226357    1.23757770    5.29688340
   -1.03893542    1.27002279    5.31442821
   -1.15158066    1.31423491    5.42212396
   -1.25546478    1.36378891    5.56501259
   -1.43689312    1.40311779    5.58369850
   -1.59395377    1.43530323    5.59219410
   -1.71960563    1.48846855    5.73386466
   -1.84957747    1.53310474    5.82639282
   -1.98261491    1.59724962    6.01278301
   -2.10355589    1.66049974    6.20857656
   -2.22678443    1.72194503    6.39282066
   -2.41215483    1.78544826    6.52743079
   -2.60115971    1.84735243    6.64988004
   -2.79899430    1.91670455    6.80404286
   -3.04308641    1.98082309    6.88777128
   -3.26893943    2.06853093    7.09963878
   -3.54662968    2.15893133    7.27398265
   -3.90655050    2.23743275    7.30971725];
elseif k == 6;
 beta=[
    0.10268529    1.06233323    6.45537060
   -0.10711788    1.08569846    6.39021589
   -0.25670243    1.11453233    6.41234545
   -0.42860607    1.13991374    6.39405029
   -0.58140487    1.17221921    6.43220656
   -0.74364084    1.20918963    6.49082631
   -0.85372368    1.25155320    6.63493421
   -1.04744667    1.28104668    6.62185275
   -1.17563497    1.32847634    6.77661103
   -1.32630035    1.36073173    6.81868665
   -1.51263821    1.39642256    6.84983688
   -1.69665826    1.43908537    6.92357226
   -1.85887381    1.48165344    7.01973450
   -2.02647435    1.53748854    7.18892690
   -2.21213866    1.57814798    7.24896540
   -2.42608039    1.62418086    7.31301399
   -2.64361156    1.66825057    7.36124314
   -2.86782588    1.70991412    7.39100489
   -3.04482252    1.78380188    7.65639362
   -3.29682525    1.84042759    7.74747106
   -3.53200103    1.91204036    7.94386865
   -3.77856039    1.99711396    8.20699916
   -4.09198662    2.07308763    8.35278953
   -4.46127051    2.15283209    8.46322975
   -4.84663661    2.26163182    8.73079979];
elseif k == 7;
 beta=[
    0.01520367    1.02127880    7.16936642
   -0.04740807    1.06481218    7.40254976
   -0.28082728    1.08452735    7.30854007
   -0.56482793    1.09568929    7.10793907
   -0.85263569    1.10438224    6.88716743
   -1.11083694    1.11973790    6.74147024
   -1.33809740    1.13527127    6.62745930
   -1.50025031    1.17145958    6.71772857
   -1.74667595    1.18765723    6.58929087
   -1.91731568    1.21740031    6.62778123
   -2.05209462    1.25979699    6.78640417
   -2.30504188    1.28596906    6.72413879
   -2.41126392    1.33985346    6.99080465
   -2.55580188    1.39540306    7.23081238
   -2.69059635    1.44733272    7.45685776
   -2.86843709    1.50525209    7.68271888
   -3.04931860    1.56365581    7.91034464
   -3.23598337    1.62633971    8.15978616
   -3.47421705    1.69402981    8.39625689
   -3.75143997    1.74247531    8.46069843
   -3.85508404    1.86136397    9.17732284
   -4.17922063    1.92750376    9.31805976
   -4.45003013    2.02239323    9.70938177
   -4.83240742    2.11485118    9.97553714
   -5.29820248    2.21309836   10.19549317];
elseif k == 8;
 beta=[
   -0.39922247    0.96614268    7.36308639
   -0.64781040    0.98829381    7.29562412
   -0.90512466    1.00600781    7.18043625
   -1.03504484    1.03978883    7.32057971
   -1.22893065    1.07002212    7.37047853
   -1.43398855    1.09930195    7.40209013
   -1.67092025    1.12422199    7.36924412
   -1.84133808    1.15985129    7.48535954
   -1.99791297    1.20320583    7.67295564
   -2.29080531    1.22047939    7.52333645
   -2.56486258    1.23983004    7.40823268
   -2.74749831    1.28319784    7.57233093
   -2.95497357    1.31812527    7.64538242
   -3.15215473    1.36269115    7.80268567
   -3.28785164    1.41800597    8.10359247
   -3.47063547    1.47886995    8.40424354
   -3.64325327    1.54274521    8.73679992
   -3.83972790    1.60188327    9.01120228
   -4.03144405    1.68070546    9.44549071
   -4.21368254    1.76777472    9.95525836
   -4.53723736    1.83612375   10.18196096
   -4.90499940    1.90675150   10.37909127
   -5.26566882    1.99132890   10.69218120
   -5.68488627    2.08232840   10.99868842
   -6.24404744    2.16669305   11.11840161];
elseif k == 9;
 beta=[
   -0.41829730    0.98199477    8.41536544
   -0.52239398    1.02190768    8.66034589
   -0.78385774    1.04090641    8.57102608
   -1.05679171    1.06612179    8.52746023
   -1.13431053    1.11379382    8.87141035
   -1.46408321    1.12917071    8.68249257
   -1.69201722    1.15810526    8.71686766
   -1.76532435    1.21077909    9.11212246
   -1.93332456    1.25495090    9.33682998
   -2.09358946    1.29879793    9.56649393
   -2.36819331    1.32795931    9.55620647
   -2.56829552    1.37686566    9.79386203
   -2.80435905    1.41167936    9.87362481
   -3.09964865    1.44716350    9.90043545
   -3.31413719    1.49894269   10.14957023
   -3.55649689    1.56267649   10.47679479
   -3.96065300    1.59085622   10.33687387
   -4.34416270    1.61822423   10.21127218
   -4.69797263    1.66561890   10.29055335
   -4.89563533    1.76081648   10.94395394
   -5.29569360    1.81416887   11.02617946
   -5.63815024    1.89211154   11.38581347
   -6.17452812    1.94475562   11.33000389
   -6.72358743    2.01453093   11.41228890
   -7.39459484    2.07256382   11.27122598];
elseif k == 10;
 beta=[
   -0.42673415    0.97696733    9.33690428
   -0.52615735    1.01469201    9.60827282
   -0.75032237    1.04137592    9.65299243
   -1.06682600    1.06035216    9.52601634
   -1.33665656    1.08890237    9.54192855
   -1.42595202    1.13690291    9.92415187
   -1.54428847    1.18401286   10.27269265
   -1.77419500    1.21937768   10.39457986
   -2.01883134    1.25721765   10.52762705
   -2.18101615    1.29626061   10.75479155
   -2.42472835    1.33886236   10.93826431
   -2.76081331    1.36952752   10.91103475
   -3.19665524    1.38218364   10.61182834
   -3.48472406    1.42443204   10.74491553
   -3.76395652    1.46501992   10.87469563
   -4.05576188    1.51664606   11.10004119
   -4.42481092    1.55053638   11.07442835
   -4.65414019    1.61302731   11.46898489
   -4.94599421    1.68182770   11.86485272
   -5.37202976    1.72713869   11.89530343
   -5.70452096    1.80670522   12.35643597
   -6.12774625    1.87590968   12.62535212
   -6.65186326    1.94208602   12.76762100
   -7.20850445    2.02442031   13.03403881
   -7.88212847    2.11313374   13.25158695];
elseif k == 11;
 beta=[
   -0.19535902    1.00559129   10.87038879
   -0.75283110    1.01038095   10.37302517
   -1.21180473    1.02016732   10.03075566
   -1.46670398    1.05491036   10.15927363
   -1.74697202    1.08478989   10.20279774
   -1.80567466    1.14132695   10.75622630
   -2.10745844    1.17186987   10.79563540
   -2.36834954    1.20501771   10.90229685
   -2.66098160    1.24282553   11.02705458
   -2.93005645    1.27498600   11.11239068
   -3.21495136    1.30574340   11.16987898
   -3.54778906    1.34177906   11.23710382
   -3.71696283    1.38722752   11.56435040
   -3.74577468    1.46849164   12.41150376
   -3.98532412    1.51818333   12.71447825
   -4.24047003    1.57846161   13.11845679
   -4.55902586    1.63036351   13.36978139
   -4.73778196    1.70097923   13.95942790
   -5.11177593    1.76039503   14.23885326
   -5.56410423    1.81295488   14.36785692
   -6.11302269    1.85389337   14.27618096
   -6.59009692    1.92606235   14.59683401
   -7.24961896    1.98408398   14.58548058
   -7.77410956    2.07779737   15.08677627
   -8.44670592    2.17452086   15.47653172];
elseif k == 12;
 beta=[
    0.62363950    1.07319721   13.46696427
    0.39480701    1.10660837   13.63758906
    0.39878595    1.16233939   14.29837849
   -0.16405274    1.17371553   13.87540451
   -0.50859206    1.20634874   13.92678409
   -0.72398492    1.24684749   14.19287607
   -1.08816368    1.27744775   14.19285263
   -1.44422212    1.31098081   14.23860130
   -1.76256188    1.34778305   14.36079309
   -2.04664975    1.38565639   14.53529802
   -2.64047790    1.39613015   14.07879262
   -3.16295762    1.41727511   13.81866487
   -3.44533971    1.45388739   13.97699217
   -3.83762830    1.49367624   14.06476906
   -4.12353121    1.54221915   14.35905377
   -4.55070001    1.58489880   14.44353335
   -5.03290868    1.61842830   14.36772826
   -5.48212779    1.66017629   14.42392278
   -5.94349081    1.71291744   14.59925758
   -6.35805907    1.77635580   14.94725292
   -6.85206179    1.83641566   15.17644294
   -7.30312143    1.91542788   15.67021099
   -7.93893685    1.98368372   15.85627296
   -8.59728756    2.06457462   16.16938142
   -9.40614589    2.15113101   16.40074311];
elseif k == 13;
 beta=[
   -0.15166327    1.03163511   13.24032273
   -0.29682393    1.07264714   13.62326086
   -0.80545944    1.09065010   13.36097820
   -1.24763888    1.11502027   13.23879029
   -1.56297598    1.14592020   13.32467925
   -1.77818824    1.18376477   13.59780088
   -2.22754573    1.20466518   13.42237457
   -2.69342304    1.21966155   13.15726135
   -3.13901783    1.24573680   13.05420910
   -3.39546712    1.28686728   13.33326049
   -3.76755467    1.32068892   13.39902442
   -4.10323327    1.36526527   13.64244660
   -4.47055214    1.39894438   13.71634202
   -4.87294776    1.44074018   13.86237247
   -5.22898241    1.48046384   14.02472973
   -5.60200180    1.53114812   14.31307228
   -5.99475933    1.57850827   14.53746550
   -6.37916131    1.62875976   14.80524171
   -6.62554324    1.72010075   15.73771480
   -7.13000793    1.77447076   15.94455535
   -7.67620371    1.83823988   16.23354516
   -8.31523358    1.89643413   16.35598098
   -8.93129759    1.97172851   16.71941790
   -9.63108026    2.06158149   17.18770184
  -10.43001290    2.15951031   17.65972012];
elseif k == 14;
 beta=[
   -0.74614558    0.98466994   13.03446479
   -1.12407592    1.01132860   13.02866703
   -1.38902163    1.04186614   13.18332311
   -1.75040170    1.07046599   13.22121524
   -2.13171585    1.08946300   13.10819662
   -2.46863371    1.11843485   13.17729036
   -2.71008800    1.15204135   13.40170002
   -3.18337613    1.17449007   13.25546195
   -3.32187073    1.23359369   13.93222399
   -3.43283866    1.29289375   14.64350519
   -3.53440241    1.35232510   15.36575032
   -3.97908760    1.38944067   15.44183772
   -4.34672029    1.42822538   15.61693107
   -4.88635260    1.45290503   15.43256322
   -5.16415211    1.49923165   15.80109236
   -5.65657182    1.53480748   15.80536528
   -6.35356208    1.54514446   15.26898366
   -6.89858142    1.57677689   15.17604839
   -7.50776978    1.59949571   14.89302915
   -8.17296699    1.62052891   14.53071223
   -8.74044583    1.67384722   14.71091158
   -9.25435034    1.75173591   15.28179093
   -9.89460165    1.81988219   15.59493229
  -10.53730211    1.91555440   16.28860418
  -11.32182992    2.02441584   17.02353770];
elseif k == 15;
 beta=[
    0.08890107    1.04605206   15.75883189
   -0.41283640    1.06936598   15.60927567
   -0.68874279    1.11694005   16.04305106
   -1.29444725    1.13489992   15.71438764
   -1.76327990    1.16209433   15.65797722
   -2.17324543    1.19354721   15.72137838
   -2.65262755    1.22023190   15.64327131
   -2.97505580    1.25708802   15.87165019
   -3.51330031    1.28140106   15.70012090
   -4.13360330    1.28632712   15.16154224
   -4.52137134    1.32077809   15.29219233
   -4.86044605    1.37195534   15.71600338
   -5.21290478    1.41807217   16.05402348
   -5.55254238    1.47308118   16.53626421
   -5.88753393    1.52208627   16.93486908
   -6.29500456    1.57263764   17.28942552
   -6.80959288    1.61301977   17.38054304
   -7.18783321    1.67231913   17.89013723
   -7.72025160    1.73633051   18.31885201
   -8.36087170    1.78043054   18.34474038
   -8.88610457    1.84691771   18.81049923
   -9.48155658    1.92222536   19.34086381
  -10.36260328    1.96752064   19.14034549
  -11.11496456    2.05563286   19.70145426
  -12.00843810    2.16528388   20.44799459];
elseif k == 16;
 beta=[
    0.46528978    1.05112529   17.25488211
    0.43008154    1.10351062   18.05053033
    0.07697252    1.13860184   18.26027314
   -0.37950973    1.16796658   18.27404649
   -0.88386002    1.19333026   18.18099203
   -1.33058369    1.22542139   18.25127443
   -1.61922129    1.27035729   18.67528843
   -1.87632670    1.31420796   19.12033097
   -1.97338603    1.37603638   20.00857771
   -2.58224544    1.39868495   19.76886902
   -2.84090858    1.45124132   20.34344564
   -2.98796550    1.51692859   21.24523244
   -3.23984964    1.56688525   21.78280095
   -3.48443482    1.63680741   22.65271599
   -3.98757090    1.67937664   22.83257296
   -4.28747788    1.75199890   23.69068270
   -4.88069483    1.80707543   23.98008544
   -5.34249111    1.86691377   24.47432402
   -5.56129322    1.96501259   25.81734211
   -6.26979326    2.02530811   26.07471515
   -7.32166933    2.06845898   25.72806144
   -8.08670318    2.14897186   26.25556111
   -8.65873574    2.25249876   27.33758012
   -9.24533172    2.38101549   28.80098978
  -10.48180252    2.46608487   28.92682924];
elseif k == 17;
 beta=[
   -0.12141974    1.03076923   17.38536404
   -0.74566631    1.04877034   17.07812936
   -0.95561978    1.09036086   17.57356401
   -0.92481394    1.14940866   18.58805077
   -1.06979100    1.19881546   19.27366690
   -1.35655478    1.24461144   19.76180159
   -1.64450229    1.28780204   20.20215689
   -1.74659172    1.34388438   21.04475621
   -1.65103495    1.41926069   22.41269941
   -1.76124366    1.47627760   23.26739057
   -2.34478192    1.50709561   23.21475569
   -3.49371502    1.50655735   22.07577551
   -4.39138552    1.51540960   21.34494784
   -5.26345745    1.53115958   20.75661649
   -5.78066359    1.56502984   20.81951981
   -6.35484599    1.60955103   21.00191306
   -6.76551924    1.66979149   21.61064515
   -7.43941487    1.70538542   21.54293558
   -7.79743512    1.79327489   22.67282154
   -8.59279577    1.83346776   22.56708577
   -9.41138447    1.88158599   22.56698715
  -10.15777121    1.95127500   23.00386795
  -10.95254487    2.02981087   23.53825850
  -11.83466650    2.13059591   24.36844647
  -12.92417615    2.22328230   24.85270129];
elseif k == 18;
 beta=[
   -0.77986818    0.98133747   16.85727760
   -0.79441135    1.02968238   17.70136645
   -1.25033389    1.05806807   17.75529080
   -1.72843077    1.08532626   17.77045115
   -2.28883097    1.10718808   17.60839004
   -2.61827901    1.14593448   17.97774096
   -3.03443573    1.18145108   18.20127152
   -3.51763928    1.21158057   18.25977507
   -3.86608691    1.25863393   18.75760151
   -4.41867669    1.28118544   18.61341726
   -4.98809197    1.31087784   18.57739926
   -5.65300187    1.33650420   18.38441051
   -5.91902494    1.39355795   19.13916085
   -6.24971747    1.45044333   19.82655948
   -6.74656405    1.48871147   20.01963099
   -7.34463122    1.53047716   20.17003888
   -8.18555334    1.54743864   19.64659922
   -8.82018605    1.58028163   19.60563813
   -9.12768124    1.67019684   20.90587060
   -9.38763893    1.77429283   22.50436080
   -9.69595478    1.88300634   24.13481943
  -10.16801544    1.99000979   25.57669172
  -10.93868504    2.08783090   26.56626751
  -12.17460856    2.14372256   26.34258679
  -13.31060245    2.24707659   27.05905240];
elseif k == 19;
 beta=[
   -0.65042637    0.98163369   18.02644062
   -1.81831855    0.97399920   16.73753537
   -1.94278557    1.02159918   17.50783638
   -2.51609942    1.04093618   17.29811107
   -3.10420030    1.06504599   17.16939322
   -3.30855946    1.11892727   17.98186994
   -3.69894784    1.15696029   18.31351373
   -4.37111787    1.17201776   17.93402304
   -4.85425504    1.20415757   18.06336817
   -5.17745594    1.24852026   18.58094026
   -5.37490214    1.30183529   19.39573218
   -5.56015152    1.36263725   20.36877257
   -5.68880549    1.43122589   21.53520800
   -6.31502145    1.46826225   21.61430918
   -7.04069583    1.49003175   21.30872332
   -7.73303975    1.52590997   21.30627214
   -8.06915474    1.59268669   22.23692063
   -8.29524268    1.67513289   23.57079963
   -8.49637905    1.77700613   25.29113390
   -8.82961560    1.86822887   26.68414873
   -9.54386980    1.94041693   27.34538265
   -9.95161969    2.05629471   29.12832021
  -10.52922214    2.17360470   30.77194400
  -12.13707536    2.22527381   30.16778942
  -13.01807866    2.36222434   31.88331065];
elseif k == 20;
 beta=[
   -1.06857636    0.96487189   18.26725099
   -1.49425447    0.99822735   18.50070479
   -1.74337241    1.04022824   19.08073059
   -2.75170577    1.03703821   18.02811531
   -3.50661636    1.05074628   17.55334104
   -4.24140268    1.06012732   17.01281401
   -4.57604636    1.09944396   17.45922072
   -5.05290626    1.12849790   17.56387804
   -5.30081522    1.17721547   18.27821840
   -5.54329536    1.22152635   18.91313597
   -5.89372773    1.26550458   19.43930795
   -6.54613046    1.29053943   19.29498906
   -6.94804549    1.32686828   19.62114687
   -7.42345994    1.36410937   19.88912757
   -8.10465313    1.37876456   19.50125692
   -8.78978243    1.41638114   19.57318581
   -9.19013766    1.48133749   20.46579412
   -9.68791843    1.54124032   21.16558952
   -9.92508150    1.64288605   22.94942945
  -10.67030768    1.69825485   23.31453573
  -11.29104874    1.77399047   24.20810423
  -11.96445351    1.87051655   25.46336470
  -12.65347799    1.96999500   26.75638178
  -13.25760827    2.10824378   28.90438436
  -14.40571130    2.22555523   30.09874911];
elseif k == 21;
 beta=[
    1.97297433    1.10127532   25.05480549
    1.41010264    1.13628944   25.23064090
    0.53285927    1.15979646   24.85781286
   -0.94100834    1.15143936   23.22723986
   -1.69833324    1.17062798   22.87754758
   -2.21379151    1.20687848   23.12509219
   -2.86729931    1.23280216   23.01981660
   -3.05582038    1.28598163   23.94572376
   -3.29250335    1.33872038   24.80867389
   -3.45171045    1.38750054   25.66127729
   -3.62151371    1.45131892   26.82817047
   -3.00884181    1.55769824   29.65205707
   -3.27591267    1.61411708   30.56098207
   -4.80766243    1.61326875   29.02339546
   -5.34675615    1.65917154   29.44674918
   -6.85262439    1.64929486   27.75083171
   -7.41788601    1.69936031   28.23427962
   -8.10515148    1.74504657   28.50657665
   -8.85187742    1.80569970   29.03653011
   -9.53699416    1.86854163   29.67580329
  -10.99298932    1.88133502   28.50197758
  -12.02206946    1.94354599   28.78037821
  -13.20057232    2.00426226   28.88416333
  -14.24487554    2.09515063   29.74230206
  -15.78264258    2.16886833   29.75304653];
elseif k == 22;
 beta=[
   -0.10333867    1.02044473   22.35494476
   -1.31357641    1.02333886   21.21671303
   -1.00773420    1.09180521   23.01024104
   -1.58969045    1.11587046   22.95028373
   -1.29658678    1.18838088   24.81078511
   -3.11539916    1.16008734   22.39340658
   -3.90131783    1.18071027   22.06733808
   -4.57991099    1.20867850   22.01091943
   -5.24461699    1.23911849   22.01737083
   -5.67469173    1.27893200   22.45876928
   -6.15591889    1.32046856   22.88973220
   -6.75687829    1.36053523   23.17204911
   -6.93873446    1.42618367   24.42804563
   -7.46998579    1.47485784   24.96318760
   -8.10278611    1.51194960   25.14400734
   -8.71442081    1.56622462   25.72526112
   -9.29148376    1.62065226   26.34478706
   -9.74666398    1.68871961   27.38155115
  -10.40132261    1.75647153   28.21430865
  -11.37321458    1.80740771   28.37225193
  -12.56948904    1.84602806   28.03243044
  -13.89110375    1.87414317   27.33642185
  -15.05081990    1.92905254   27.38960135
  -16.24074969    2.02447803   28.30254211
  -17.48616754    2.14047062   29.59988693];
elseif k == 23;
 beta=[
   -0.49944590    0.99922532   22.43566770
   -1.36356658    1.01329566   21.90318409
   -1.28387810    1.07326068   23.34285517
   -3.03197986    1.04848290   21.05511228
   -3.76841383    1.06740971   20.76282273
   -4.48252295    1.08694087   20.50746941
   -4.70292726    1.13205490   21.31708083
   -5.17544096    1.17072957   21.72987930
   -5.23784265    1.24108760   23.27511180
   -6.05568370    1.25708848   22.83863121
   -6.56260456    1.29757491   23.26143691
   -7.16329232    1.34258401   23.69866541
   -7.85401092    1.36819577   23.60680014
   -8.54807303    1.40682801   23.80075480
   -9.05993683    1.45443162   24.38446965
   -9.81650218    1.49524398   24.57156522
  -10.59330662    1.53132460   24.63008734
  -11.48476294    1.55320309   24.25325489
  -12.32889396    1.60506783   24.60546924
  -12.95451688    1.67852996   25.66208928
  -13.82161556    1.74527091   26.32904867
  -14.79947177    1.80891881   26.81267116
  -16.08208345    1.85011921   26.48270157
  -17.61237247    1.87211318   25.46902265
  -19.12881744    1.93892537   25.48896114];
elseif k == 24;
 beta=[
   -0.07382888    1.02271887   24.43680134
   -1.26322734    1.03382223   23.52458049
   -2.60490334    1.02874967   22.07674673
   -3.48119599    1.04345033   21.55644109
   -3.55498110    1.09869139   22.79699871
   -4.34385560    1.11606845   22.43173039
   -5.11563904    1.13513926   22.12063636
   -5.22861690    1.19589374   23.46547247
   -5.83883708    1.22980280   23.67117704
   -6.14841890    1.27492803   24.43688745
   -6.75892171    1.30936942   24.65760295
   -7.40239404    1.35007846   24.99271404
   -8.04113095    1.38319446   25.15653114
   -8.88996224    1.41237037   25.01373004
   -9.60277155    1.45031666   25.21699769
  -10.23553244    1.50475169   25.88811191
  -11.15516423    1.53534232   25.70117140
  -11.74054967    1.59282089   26.48792651
  -12.64249117    1.64075487   26.73601663
  -13.30058852    1.71217075   27.79007603
  -14.36970020    1.75684854   27.79723761
  -15.40563376    1.82266468   28.34490487
  -16.73637622    1.87057718   28.16697139
  -17.85760602    1.95847387   29.15013237
  -19.36728421    2.04371428   29.68735594];
elseif k == 25;
 beta=[
   -1.46743340    0.97072863   22.81467938
   -2.40598835    0.98424934   22.22678235
   -2.85067148    1.01744165   22.61351649
   -2.54300447    1.09146496   24.75285236
   -3.94854646    1.08422762   23.18114119
   -4.75529934    1.10330284   22.85835485
   -5.06488584    1.15101041   23.73278045
   -5.88418659    1.16718182   23.32167885
   -6.65851188    1.18182539   22.91510965
   -7.23621089    1.21258115   23.10825675
   -7.32961013    1.28222425   24.74394947
   -8.31094419    1.29755319   24.15828568
   -9.03422294    1.32582403   24.14836168
   -9.47887387    1.37954112   25.03874167
  -10.26028624    1.40908698   25.00211611
  -10.98194471    1.44979796   25.29924252
  -11.36270788    1.52744411   26.84723386
  -11.79284026    1.59552523   28.11099139
  -12.25407809    1.68278661   29.81926578
  -13.12218859    1.73971629   30.37362247
  -14.23263315    1.78846130   30.48691574
  -15.42844969    1.83484532   30.45422383
  -16.13660857    1.94882338   32.58057276
  -17.51565822    2.02887672   33.20082807
  -19.32818091    2.10293558   33.24531145];
elseif k == 26;
 beta=[
   -0.95376699    0.99010068   24.78768335
   -0.30618874    1.06819438   27.44718359
   -0.84058713    1.10481920   27.86927671
   -1.57072315    1.13661824   27.96995781
   -1.98400630    1.18098260   28.70308432
   -2.82265301    1.20637220   28.52407763
   -4.00594213    1.21332095   27.52688448
   -5.64523993    1.20063832   25.57476578
   -5.94946748    1.25417905   26.65870689
   -6.71397908    1.27963732   26.56405688
   -7.66978944    1.29922339   26.12394224
   -8.65959083    1.31235441   25.48544047
   -9.25031883    1.35106842   25.90057251
   -9.63632239    1.41186703   27.08716695
   -9.62934669    1.49587189   29.26180114
  -10.14565063    1.56186682   30.45811427
  -10.98947597    1.60910469   30.84475685
  -11.76168667    1.65136564   31.17193740
  -12.89351651    1.69057609   31.06673018
  -13.84477150    1.74733590   31.59885641
  -14.65311637    1.83137222   32.97296517
  -15.44071172    1.92973380   34.74054444
  -16.72436804    2.00489849   35.41469519
  -18.75440528    2.03507571   34.18651084
  -19.84703601    2.19185630   37.15811931];
elseif k == 27;
 beta=[
   -0.67130408    1.00332759   26.44683813
   -1.48083119    1.02362498   26.17512096
   -2.73111905    1.02976753   25.10339631
   -3.36786588    1.05902898   25.25543006
   -4.26921053    1.07791814   24.86895727
   -5.25167535    1.09239424   24.28530031
   -6.06883493    1.11611455   24.10616090
   -6.84942139    1.13383909   23.80646679
   -7.76203152    1.15176049   23.38083302
   -8.44975761    1.18073433   23.47934425
   -9.00958089    1.22295863   24.05882773
   -9.07739717    1.30430570   26.17041119
   -9.63204180    1.35473123   26.98019551
  -10.48258559    1.38446900   26.93823076
  -10.87560163    1.44879586   28.27685356
  -11.50329385    1.50519723   29.17089274
  -12.19655550    1.56047904   29.96763938
  -13.02270052    1.60803407   30.42626268
  -13.69121688    1.69089020   31.99647362
  -14.11842914    1.78912285   34.20279743
  -15.39872829    1.83650964   34.20801567
  -16.63914317    1.89839544   34.64202021
  -17.91300325    1.96406233   35.13747575
  -19.45151952    2.03986034   35.64202940
  -21.24088536    2.12898496   36.25924742];
elseif k == 28;
 beta=[
    0.94895902    1.06701762   30.80698696
   -0.81896279    1.05850193   28.80386776
   -1.96547406    1.07463827   28.11236876
   -2.36915604    1.12140775   29.01029582
   -2.98265134    1.15777548   29.41139700
   -4.08937683    1.16824133   28.60756837
   -5.03563527    1.18688826   28.19409163
   -6.42753782    1.18273354   26.70374189
   -7.49470921    1.19761666   26.06134083
   -8.24856050    1.22268828   26.01782748
   -8.80918764    1.26571746   26.66670059
   -9.56835317    1.29782899   26.80625628
  -10.17180392    1.33615335   27.27674815
  -11.14971916    1.36276248   27.05531346
  -11.93252903    1.39696466   27.23360086
  -12.52014150    1.46338484   28.49653476
  -13.11118312    1.52919928   29.74502191
  -13.59308795    1.60137551   31.27305544
  -14.75898528    1.64384531   31.29751170
  -15.52041701    1.72358265   32.76702958
  -16.21681776    1.81888298   34.72596923
  -16.76807582    1.93911715   37.52711111
  -17.67625126    2.05387787   39.82824915
  -19.79101551    2.10095940   39.04391698
  -21.98191970    2.16894374   38.75876145];
elseif k == 29;
 beta=[
   -1.16142041    0.97791041   27.21548272
   -1.02789522    1.03817226   29.08548408
   -1.54522437    1.07502205   29.63200928
   -2.19211308    1.10436964   29.83670504
   -2.30872497    1.16921303   31.59808605
   -2.74306832    1.21566869   32.50150626
   -3.44117926    1.25462890   32.93135408
   -4.38688823    1.27593616   32.61042179
   -6.58865502    1.24969394   29.67924317
   -8.12692041    1.23780491   27.81759331
   -7.95450591    1.31961718   30.34403294
   -8.04634485    1.39300880   32.36845772
   -8.89555276    1.42526482   32.45038417
   -9.81217532    1.45995387   32.53473198
  -10.51799215    1.50172819   33.03209354
  -10.94709191    1.57671506   34.76010651
  -11.03319096    1.67687637   37.55717707
  -11.92600974    1.72345397   38.01299930
  -13.13657490    1.76501239   38.00281224
  -14.04292857    1.82616993   38.86536548
  -15.46711150    1.86907206   38.68764801
  -16.78745800    1.93713869   39.34087483
  -18.27749503    2.01397799   40.08741842
  -20.11200214    2.08954003   40.44965418
  -21.84222901    2.19671735   41.81573878];
elseif k == 30;
 beta=[
   -0.43819640    1.02502674   30.29798809
   -1.47029120    1.04430011   29.85174531
   -2.27305857    1.07158893   29.87059541
   -1.91772233    1.13956070   32.24411317
   -2.48464154    1.17310862   32.68128619
   -2.80480672    1.22323365   33.85319783
   -2.92862552    1.27990071   35.42484618
   -2.41108337    1.36285607   38.40828239
   -5.36686921    1.32357437   34.30770870
   -6.77588983    1.32821954   33.05112827
   -7.97922171    1.34609820   32.39371055
   -9.24465185    1.36132148   31.58928871
  -10.23841361    1.39055680   31.47978942
  -10.92081412    1.44738039   32.49766129
  -11.86741145    1.48087846   32.55751066
  -12.73282629    1.53704469   33.37571566
  -13.58354654    1.59154973   34.15886806
  -14.53323186    1.64060604   34.68242889
  -16.17641148    1.65696447   33.53377538
  -17.26816956    1.72132247   34.37058355
  -17.58890868    1.85179547   37.94539854
  -17.44617398    2.01690529   43.00111181
  -19.98137929    2.01439329   40.41541575
  -21.98791179    2.06361423   39.89314701
  -24.32078928    2.11397301   39.07725209];
elseif k == 31;
 beta=[
   -1.67460717    0.97764691   28.62891058
   -3.40760675    0.97035959   26.69299282
   -4.13212423    0.99715067   26.80860374
   -3.99589796    1.05893122   28.84005267
   -4.08097235    1.10996625   30.32309917
   -4.42412521    1.16457994   31.65931260
   -4.34823458    1.23251472   33.83530688
   -4.20761822    1.30200503   36.11252530
   -5.27080372    1.33644344   36.12812288
   -6.47964857    1.35596092   35.52823582
   -7.80623863    1.37465318   34.78401086
   -9.01086151    1.40068669   34.39000476
   -9.65875000    1.44520110   35.12242874
  -10.68283422    1.48478437   35.32816114
  -11.76661588    1.51120166   35.06699480
  -12.79602647    1.55112409   35.28098398
  -13.76017013    1.59787118   35.77134091
  -15.02526774    1.62612497   35.38895633
  -15.41345181    1.72464025   38.03691022
  -16.74895734    1.77317788   38.21142625
  -18.01717057    1.83663123   38.91181547
  -19.32515349    1.89895157   39.53777825
  -20.07862869    2.03730727   43.05957665
  -22.24272363    2.08953253   42.51813463
  -24.64165723    2.15192026   42.05772005];
elseif k == 32;
 beta=[
   -0.37989117    1.00396176   31.70693470
   -1.95767177    1.00771302   30.25868243
   -2.99420262    1.02868897   29.89926728
   -4.13462541    1.04563726   29.31021055
   -4.82814179    1.07623317   29.59795573
   -5.43466500    1.11346324   30.18111524
   -6.40989249    1.13657194   29.95231243
   -6.95704655    1.17947464   30.77245445
   -7.33083093    1.23803453   32.26059799
   -8.06084596    1.27233571   32.62818056
   -8.94026927    1.30337311   32.74119682
   -9.61306306    1.34872151   33.51751088
   -9.61486195    1.42991595   36.10521319
  -10.65798045    1.46906263   36.31872531
  -11.75349631    1.49517797   36.06259405
  -13.00028046    1.53051888   35.95254528
  -14.71322632    1.53690456   34.46108374
  -16.11952232    1.54552714   33.34499609
  -16.96184693    1.61676685   34.78207497
  -17.70644415    1.68773193   36.29443198
  -19.02561913    1.73587190   36.51641032
  -20.63279929    1.78432291   36.47270415
  -22.08313380    1.85183001   37.18188251
  -23.95861148    1.90780945   37.10646033
  -26.24712531    1.94800549   36.11460313];
elseif k == 33;
 beta=[
    0.57804100    1.04529153   35.08307012
    0.05122115    1.08483932   35.85506656
   -1.05825025    1.11061605   35.59003538
   -1.81088585    1.14601833   36.00902584
   -3.30385885    1.15628718   34.85644558
   -3.81223056    1.20004638   35.79082424
   -4.30958028    1.24637184   36.81914491
   -5.36837609    1.27197905   36.61057379
   -6.25788263    1.31185805   37.03447703
   -7.12183877    1.34540667   37.27213702
   -7.62064397    1.39625673   38.44423117
   -8.58498372    1.43888693   38.88626333
   -9.19507383    1.48734687   39.86901650
  -10.95974053    1.49514711   38.36964892
  -11.30956997    1.56113764   40.18492565
  -11.42447736    1.65849042   43.26825750
  -11.95554829    1.73603050   45.28693221
  -12.76888286    1.78617953   46.11688715
  -13.10861832    1.88403989   48.99337703
  -14.66299742    1.93146345   49.01277518
  -15.80153057    2.00729117   50.37950622
  -17.58628066    2.07115685   50.70882209
  -19.06482190    2.16058338   52.17634081
  -21.24659237    2.22662655   52.17806046
  -23.68246701    2.31691351   52.72584328];
elseif k == 34;
 beta=[
    0.56182534    1.05467198   36.43149734
    2.30224634    1.15131705   41.42715748
    2.33329203    1.20266057   43.19553684
    0.93244712    1.21927897   42.36419775
   -2.81777123    1.16565298   36.82445367
   -3.58613158    1.19465814   37.04036689
   -5.60296411    1.18308730   34.63333401
   -8.36302809    1.13316340   30.20536445
   -8.86450406    1.18111669   31.32099173
   -9.00664855    1.23782268   33.08682086
   -9.91638753    1.27191278   33.33377412
  -11.16388994    1.28899794   32.66998704
  -12.60594910    1.29286566   31.36962606
  -13.85299615    1.31080314   30.73273874
  -14.62956158    1.35560637   31.47451274
  -15.79520984    1.38751275   31.39693733
  -16.94177944    1.41884249   31.32076886
  -18.05325433    1.45006589   31.27117381
  -19.11190056    1.50474967   32.07095827
  -20.48496845    1.54010735   31.91199775
  -21.68503430    1.59694222   32.63974507
  -23.05552418    1.64685788   32.96751075
  -24.50892293    1.71351590   33.78219473
  -26.20583521    1.79250743   34.77021363
  -28.29264587    1.87296393   35.42136152];
elseif k == 35;
 beta=[
    0.68221505    1.05194994   37.36687278
    0.17009469    1.09166188   38.23869990
   -0.76158587    1.11826825   38.23933631
   -3.26923227    1.09923551   35.08815853
   -2.67673906    1.17772189   38.40750725
   -2.96167774    1.23267290   40.03823145
   -3.62747681    1.27892514   40.98914801
   -4.38307298    1.32838899   41.97436879
   -4.86699878    1.38760782   43.56676979
   -4.78346397    1.45454140   45.98321789
   -6.01519748    1.48431482   45.79707609
   -7.72742427    1.50223196   44.72336204
   -8.59952063    1.54552951   45.37146687
  -10.29195157    1.56785481   44.47630692
  -10.69106889    1.63361925   46.36805176
  -11.42980044    1.69541856   47.79016695
  -14.39240474    1.65767350   43.53719967
  -16.38267413    1.65735546   41.55601060
  -18.14680347    1.68195898   40.66995634
  -19.29558298    1.74376744   41.68354251
  -20.03489594    1.84448742   44.45600709
  -21.13863286    1.93565541   46.53366351
  -22.84612627    2.00785842   47.34961972
  -24.51505512    2.10960064   49.24161917
  -26.94321346    2.19433912   49.77894595];
elseif k == 36;
 beta=[
   -1.35472954    0.99030163   34.24007988
   -3.06845258    0.98831467   32.46674080
   -3.90084719    1.01583015   32.62047331
   -4.34947221    1.05853149   33.69719778
   -4.19485297    1.12117056   36.08774049
   -3.90397309    1.19372632   38.97434698
   -4.25013662    1.24916440   40.62280918
   -5.81908457    1.26546520   39.66163393
   -8.25914549    1.24974154   36.68469843
   -9.26659831    1.28219917   36.85161594
  -10.79545059    1.29247743   35.70143308
  -12.04781726    1.32021308   35.45609346
  -12.90038928    1.36072271   36.06199420
  -14.04357897    1.39991339   36.33172811
  -15.09323979    1.43646648   36.60161816
  -16.04969718    1.49030180   37.57573496
  -17.49624440    1.51694891   37.09461048
  -18.77514898    1.54984335   37.00518484
  -20.42999621    1.57739807   36.35277517
  -21.86181448    1.62179799   36.52339032
  -23.06208923    1.70769939   38.41532706
  -24.48195761    1.77821751   39.53443346
  -26.41630888    1.81767698   39.03067507
  -28.49189181    1.86837522   38.78878304
  -30.64957123    1.94133097   39.26029959];
elseif k == 37;
 beta=[
   -1.38051785    0.99879680   35.61811345
   -0.46390982    1.07878364   39.47099341
   -0.74683015    1.12523276   40.89312713
   -0.36103151    1.18972091   43.64307886
   -0.45284135    1.24748957   45.68198293
   -2.50261940    1.25896122   44.08064692
   -3.38183348    1.29771322   44.63930964
   -1.33227742    1.41629807   51.04610334
   -1.97489927    1.46713279   52.28064694
   -4.27939630    1.46266446   49.82237186
   -6.35268879    1.46739423   47.93962943
   -8.15593274    1.48434084   46.77697023
   -8.94711754    1.53128246   47.71423680
  -10.13698874    1.57893522   48.28686634
  -11.52600095    1.61260901   48.14379737
  -12.84707931    1.65361925   48.33860753
  -14.28937769    1.70014900   48.62748619
  -15.45108658    1.75118608   49.35479177
  -17.14392342    1.79931776   49.44660200
  -18.27605937    1.87857401   51.24601202
  -19.86169978    1.93613245   51.79312212
  -21.56639473    2.00556409   52.65653089
  -23.62525752    2.07109766   53.02273498
  -25.10154041    2.19522589   56.12889209
  -27.45719434    2.31348632   58.14570530];
elseif k == 38;
 beta=[
   -0.53233662    1.00814555   37.87671254
   -1.65368201    1.03084904   37.62014299
   -1.28591260    1.09779299   40.51422388
   -3.44674105    1.09746795   38.35706231
   -4.99468762    1.11056146   37.31947917
   -5.89903524    1.14272994   37.63781562
   -7.27249154    1.16239290   37.01409799
   -8.76646332    1.17069642   35.83702613
  -10.04683572    1.19610180   35.52942267
  -11.22067037    1.21426864   35.04807469
  -12.30200683    1.23714405   34.83214592
  -12.96917130    1.29191503   36.23469020
  -14.31444806    1.30514091   35.39971684
  -15.69330647    1.32130240   34.63672253
  -16.85519953    1.34696077   34.45431376
  -17.85756779    1.38577276   34.92527142
  -19.24751909    1.40473260   34.26032639
  -20.19338281    1.45403575   35.18157368
  -21.34117371    1.50734873   36.05826289
  -22.70845720    1.55384489   36.46217302
  -24.47467145    1.58802107   36.00474123
  -26.10379055    1.63500845   36.16133898
  -27.55180902    1.72296703   38.05387107
  -29.32636872    1.82056274   39.98324299
  -31.72752332    1.86453375   39.25468738];
elseif k == 39;
 beta=[
   -0.75173212    0.99953735   38.20929588
   -2.51340314    1.00591238   36.71310418
   -2.80419749    1.05020437   38.13578472
   -2.78868035    1.10359635   40.22641658
   -3.43270126    1.14433481   41.17029081
   -4.56022319    1.17355804   41.18731651
   -5.25424207    1.21393820   42.06647286
   -6.18981545    1.24847831   42.48760375
   -6.94108034    1.29609779   43.58944339
   -7.97542270    1.32863267   43.83175078
   -8.54858876    1.38110878   45.30981735
   -9.71446422    1.42349920   45.79987488
  -11.22633071    1.44135083   44.99455113
  -12.69119529    1.47449832   44.83357448
  -13.54298483    1.51829845   45.69299055
  -13.69399282    1.61679247   49.37472620
  -14.65322460    1.67871774   50.83181565
  -15.59381531    1.73995835   52.27707999
  -16.84622271    1.80974016   53.74892714
  -17.84147948    1.89122238   55.92673285
  -18.62106614    1.99283676   59.10678519
  -20.94262558    2.05003448   59.02503659
  -23.03711649    2.12352955   59.80296980
  -25.17612643    2.21541462   61.25030096
  -28.30006912    2.28194892   60.72334395];
elseif k == 40;
 beta=[
    0.37993021    1.03609179   41.78827518
    1.38675186    1.11185378   45.80074024
    0.09757811    1.13396017   45.38210374
   -1.11490757    1.15956656   45.19219128
   -1.75174293    1.20170658   46.22623181
   -1.31404368    1.26949663   49.34860001
   -2.30919855    1.30323532   49.69320477
   -3.76045742    1.33457949   49.49202238
   -6.20254253    1.33150051   46.94139342
   -7.12014555    1.36602147   47.40931509
   -8.54601009    1.39112200   46.98953209
  -10.07357853    1.41956190   46.60212337
  -12.15274573    1.41871014   44.51455686
  -13.72990082    1.44379453   43.94800720
  -15.49896840    1.45055979   42.46491535
  -17.14333904    1.46969437   41.59432522
  -18.41200621    1.50751639   41.83664551
  -19.54005143    1.56571430   43.03326316
  -21.33358514    1.57838863   41.75798611
  -22.43927623    1.65455814   43.69397450
  -24.20609680    1.69348854   43.49245560
  -25.50735052    1.78114182   45.69025648
  -27.55116336    1.83443578   45.78310880
  -29.21270670    1.94664388   48.59823783
  -31.57842409    2.04052128   49.98288587];
end;


