%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% BREAK_T.M

% This file contains the Matlab procedure  BREAK_T  written by

% Bruce E. Hansen
% Department of Economics
% Social Science Building
% University of Wisconsin
% Madison, WI 53706-1393
% bhansen@ssc.wisc.edu
% http://www.ssc.wisc.edu/~bhansen/

% The procedure implements the testing methods dicussed in
% "Testing for Structural Change in Conditional Models."

% The file calls the procedures PV_SUP, PV_EXP and PV_AVE,
% which are contained in separate ASCII files

% Format:   break_t(y,x,qvar,t1,t2);

% There are no returns from the procedure, only written output.
% Inputs:

% y    = dependent variable (nx1 vector)
% x    = regressors (nxk matrix, may contain lagged y's)
% qvar = (k+1)x1 string vector, containing names of y and x variables
%        First element contains name of y, remainder names of x's
% t1   = starting breakpoint index or percentage,
%        may be integer in [k+1,T-k-1] or percentage in (0,1).
% t2   = ending breakpoint index or percentage,
%        may be integer in [k+1,T-k-1] or percentage in (0,1),
%        must equal or exceed t1.
%        Note: For Sup test, Andrews recommends t1=.15 and t2=.85
%              For Exp test, Andrews-Ploberger recommend t1=.02 and t2=.98

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function break_t(y,x,qvar,t1,t2);
global out;
r1=10;
r2=100;
% Note: Number of Bootstrap Replications is r1*r2.
%        The program works in blocks of r2 at a time.
%        The higher is r2, the faster the program will be,
%        but the more memory is required.  
k=length(x(1,:));
n=length(y(:,1));

if 1-(n==length(x(:,1)))
    disp('ERROR:  Number of rows in Y and X must equal.');
    disp('      Please re-specify.');
end;

if t1<1
    n1=floor(n*t1);
    tau1=t1;
else
    n1=t1;
    tau1=t1/n;
end;
if t2<1
    n2=floor(n*t2);
    tau2=t2;
else
    n2=t2;
    tau2=t2/n;
end;
fprintf(out,'Dependent Variable:                 %s\n\n',qvar(1,:));
fprintf(out,'Starting Index for Break Search:    %f\n',n1);
fprintf(out,'(Percentage):                       %f\n',tau1);
fprintf(out,'Ending Index for Break Search:      %f\n',n2);
fprintf(out,'(Percentage):                       %f\n',tau2);

if n1<=k
    disp('ERROR:  Starting Sample is smaller than Number of Parameters');
    disp('        You need to select a larger value for t1');
end;
if n2>=n-k
    disp('ERROR:  Ending Sample is smaller than Number of Paramters');
    disp('        You need to select a smaller value for t2');
end;

xx=x'*x;
xxi=inv(xx);
beta=xxi*(x'*y);
e=y-x*beta;
xe=x.*(e*ones(1,length(x(1,:))));
ee=e'*e;
sig=ee/(n-k);
se=sqrt(diag(xxi)*sig);
yd=y-mean(y);
r_2=1-ee/(yd'*yd);
fprintf(out,'\n');
fprintf(out,'Full Sample Estimation\n');
fprintf(out,'Sample Size:                         %u\n',n);
fprintf(out,'Number of Regressors:                %u\n',k);
fprintf(out,'Sample Variance:                     %f\n',sig);
fprintf(out,'R-squared:                           %f\n',r_2);
fprintf(out,'\n\n');
fprintf(out,'Estimates and Standard Errors from Full Sample: \n');
for ind=1:length(beta)
     fprintf(out,'%s   %f   %f\n',qvar(ind+1,:),beta(ind),se(ind));
end;
fprintf(out,'\n\n\n');

f=zeros(n,1);
m=x(1:n1-1,:)'*x(1:n1-1,:);
mi=inv(m);
msi=inv(xx-m);
sn=sum(xe(1:n1-1,:))';

ib=n1;
while ib<=n2
    xi=x(ib,:)';
    xim=xi'*mi;
    mi=mi-((xim'*xim)'/(1+xim*xi)')';
    xim=xi'*msi;
    msi=msi+((xim'*xim)'/(1-xim*xi)')';
    sn=sn+(xe(ib,:)');
    q=sn'*msi*xx*mi*sn;
    f(ib)=q*(n-k*2)/(ee-q);
    ib=ib+1;
end;
[supf,im]=max(f);
avef=mean(f(n1:n2))';
expf=log(mean(exp(f(n1:n2)/2))');

% Standard P-Value %
l_0 = tau2*(1-tau1)/(tau1*(1-tau2));
pi_0 = 1/(1+sqrt(l_0));
pv_s = pv_sup(supf,k,l_0);
pv_e = pv_exp(expf,k,l_0);
pv_a = pv_ave(avef,k,l_0);

x1=x(1:im,:);
x2=x(im+1:n,:);
y1=y(1:im);
y2=y(im+1:n);
xx1=inv(x1'*x1);
xx2=inv(x2'*x2);
beta1=xx1*(x1'*y1);
beta2=xx2*(x2'*y2);
e1=y1-x1*beta1;
e2=y2-x2*beta2;
sig1=(e1'*e1)/(im-k);
sig2=(e2'*e2)/(n-im-k);
se1=sqrt(diag(xx1)*sig1);
se2=sqrt(diag(xx2)*sig2);
em=[e1;e2];
yd1=y1-mean(y1)';
yd2=y2-mean(y2)';
r21=1-(e1'*e1)/(yd1'*yd1);
r22=1-(e2'*e2)/(yd2'*yd2);

fprintf(out,'Estimated Breakpoint (index):      %f\n',im);
fprintf(out,'Percentage of Sample:              %f\n\n',im/n);
fprintf(out,'First Sample:\n');
fprintf(out,'Number of Observations:            %f\n',im);
fprintf(out,'Sample Variance:                   %f\n',sig1);
fprintf(out,'R-squared:                         %f\n\n\n',r21);
fprintf(out,'Estimates and Standard Errors from First Sample:\n\n');
for ind=1:length(beta)
     fprintf(out,'%s   %f   %f\n',qvar(ind+1,:),beta1(ind),se1(ind));
end;
fprintf(out,'\n\n\n');
fprintf(out,'Second Sample:\n');
fprintf(out,'Number of Observations:            %f\n',n-im);
fprintf(out,'Sample Variance:                   %f\n',sig2);
fprintf(out,'R-squared:                         %f\n\n\n',r22);
fprintf(out,'Estimates and Standard Errors from First Sample:\n\n');
for ind=1:length(beta)
     fprintf(out,'%s   %f   %f\n',qvar(ind+1,:),beta2(ind),se2(ind));
end;
fprintf(out,'\n\n\n');


% Fixed Regressor Bootstrap  %
supfb = zeros(r2,r1);
expfb = zeros(r2,r1);
avefb = zeros(r2,r1);
supfh = zeros(r2,r1);
expfh = zeros(r2,r1);
avefh = zeros(r2,r1);

ri=1;
while ri<r1
    u=normrnd(0,1,n,r2);
    euf=u-x*xxi*(x'*u);
    eef=sum(euf.^2)';
    ff=zeros(n,r2);
    snf=x(1:n1-1,:)'*euf(1:n1-1,:);
    
    uh=u.*(em*ones(1,length(u(1,:))));
    euh=uh-x*xxi*(x'*uh);
    eeh=sum(euh.^2)';
    ffh=zeros(n,r2);
    snh=x(1:n1-1,:)'*euh(1:n1-1,:);
    
    mf=x(1:n1-1,:)'*x(1:n1-1,:);
    mif=inv(mf);
    msif=inv(xx-mf);
    
    ib=n1;
    while ib<=n2
     xif=x(ib,:)';
     ximf=xif'*mif;
     mif=mif-(ximf'*ximf)/(1+ximf*xif);
     ximf=xif'*msif;
     msif=msif+(ximf'*ximf)/(1-ximf*xif);
     snf=snf+xif*euf(ib,:);
     qf=sum(snf.*(msif*xx*mif*snf))';
     ff(ib,:)=(qf./(eef-qf))';
     snh=snh+xif*euh(ib,:);
     qf=sum(snh.*(msif*xx*mif*snh))';
     ffh(ib,:)=(qf./(eeh-qf))';
     ib=ib+1;
    end;
    
    ftf=ff(n1:n2,:)*(n-k*2);
    supfb(:,ri)=(max(ftf)'>supf);
    expfb(:,ri)=(mean(exp(ftf/2))'>exp(expf));
    avefb(:,ri)=(mean(ftf)'>avef);
    
    ftf=ffh(n1:n2,:)*(n-k*2);
    supfh(:,ri)=(max(ftf)'>supf);
    expfh(:,ri)=(mean(exp(ftf/2))'>exp(expf));
    avefh(:,ri)=(mean(ftf)'>avef);
    ri=ri+1;
end;
supfb = mean(mean(supfb)');
expfb = mean(mean(expfb)');
avefb = mean(mean(avefb)');

supfh = mean(mean(supfh)');
expfh = mean(mean(expfh)');
avefh = mean(mean(avefh)');

fprintf(out,'Tests for Structural Change\n\n');
fprintf(out,'Bootstrap Replications:          %u\n\n',r1*r2);
fprintf(out,'              Test        Andrews      Bootstrap    Hetero-Corrected\n');
fprintf(out,'              Statistic   P-Value      P-Value      P-Value\n\n\n');
fprintf(out,'SupF          %f   %f   %f   %f \n',supf,pv_s,supfb,supfh);
fprintf(out,'ExpF          %f   %f   %f   %f \n',expf,pv_e,expfb,expfh);
fprintf(out,'ExpF          %f   %f   %f   %f \n\n\n',avef,pv_a,avefb,avefh);


    
    