/************************************************************************


BREAK.PRC

This file contains the GAUSS procedure  BREAK_T  written by

Bruce E. Hansen
Department of Economics
Social Science Building
University of Wisconsin
Madison, WI 53706-1393
bhansen@ssc.wisc.edu
http://www.ssc.wisc.edu/~bhansen/

The procedure implements the testing methods dicussed in
"Testing for Structural Change in Conditional Models."

The file calls the procedures PV_SUP, PV_EXP and PV_AVE,
which are contained in separate ASCII files

Format:   break_t(y,x,qvar,t1,t2);

There are no returns from the procedure, only written output.
Inputs:

y    = dependent variable (nx1 vector)
x    = regressors (nxk matrix, may contain lagged y's)
qvar = (k+1)x1 string vector, containing names of y and x variables
       First element contains name of y, remainder names of x's
t1   = starting breakpoint index or percentage,
       may be integer in [k+1,T-k-1] or percentage in (0,1).
t2   = ending breakpoint index or percentage,
       may be integer in [k+1,T-k-1] or percentage in (0,1),
       must equal or exceed t1.
       Note: For Sup test, Andrews recommends t1=.15 and t2=.85
             For Exp test, Andrews-Ploberger recommend t1=.02 and t2=.98

*************************************************************************/

#include pv_sup.prc;
#include pv_exp.prc;
#include pv_ave.prc;

proc (0) = break_t(y,x,qvar,t1,t2);
local r1,r2,k,n,mas,fmt,n1,tau1,n2,tau2,xx,xxi,beta,e,xe,ee,
sig,se,yd,r_2,yp,f,m,mi,msi,sn,ib,xi,xim,q,im,supf,expf,avef,
l_0,pi_0,pv_s,pv_e,pv_a,x1,x2,y1,y2,xx1,xx2,beta1,beta2,e1,e2,
sig1,sig2,se1,se2,em,yd1,yd2,r21,r22,supfb,expfb,avefb,supfh,
expfh,avefh,ri,u,euf,eef,ff,snf,uh,euh,eeh,ffh,snh,mf,mif,msif,
xif,ximf,ximf,qf,ftf;


r1 = 10;
r2 = 100;
@ Note: Number of Bootstrap Replications is r1*r2.
        The program works in blocks of r2 at a time.
        The higher is r2, the faster the program will be,
        but the more memory is required.  @
k = cols(x);
n = rows(y);
format 12,5;
mas = 0~1~1;
let fmt = "-*.*lG" 16 8;
if n .ne rows(x);
  "ERROR:  Number of rows in Y and X must equal.";
  "        Please re-specify.";
endif;
if t1<1;  n1 = floor(n*t1);  tau1=t1;
else;  n1=t1;  tau1=t1/n;
endif;
if t2<1;  n2 = floor(n*t2);  tau2=t2;
else;  n2=t2;  tau2=t2/n;
endif;
"Dependent Variable              = " $qvar[1];
"";
"Starting Index for Break Search = " n1;
"   (Percentage)                 = " tau1;
"Ending Index for Break Search   = " n2;
"   (Percentage)                 = " tau2;
if n1 <= k;
  "ERROR:  Starting Sample is smaller than Number of Parameters";
  "        You need to select a larger value for t1";
endif;
if n2 >= (n-k);
  "ERROR:  Ending Sample is smaller than Number of Paramters";
  "        You need to select a smaller value for t2";
endif;


  xx = moment(x,0);
  xxi = invpd(xx);
  beta = xxi*(x'y);
  e = y - x*beta;
  xe = x.*e;
  ee = e'e;
  sig = ee/(n-k);
  se = sqrt(diag(xxi)*sig);
  yd = y-meanc(y);
  r_2 = 1 - ee/(yd'yd);
  "";
  "Full Sample Estimation";
  "Sample Size                     = " n;
  "Number of Regressors            = " k;
  "Sample Variance                 = " sig;
  "R-squared                       = " r_2;
  "";"";
  "Estimates and Standard Errors from Full Sample: ";"";
  yp=printfm(qvar[2:k+1]~beta~se,mas,fmt');
  "";"";"";
  f  = zeros(n,1);
  m = moment(x[1:n1-1,.],0);
  mi = invpd(m);
  msi = invpd(xx-m);
  sn = sumc(xe[1:n1-1,.]);

    ib = n1; do while ib <= n2;
      xi = x[ib,.]';
      xim = xi'mi;
      mi = mi - (xim'xim)/(1+xim*xi);
      xim = xi'msi;
      msi = msi + (xim'xim)/(1-xim*xi);
      sn = sn + (xe[ib,.]');
      q = sn'msi*xx*mi*sn;
      f[ib] = q*(n-k*2)/(ee-q);
    ib = ib+1; endo;
    im = maxindc(f);
    supf = f[im];
    avef = meanc(f[n1:n2]);
    expf = ln(meanc(exp(f[n1:n2]/2)));

  @ Standard P-Value @
  l_0 = tau2*(1-tau1)/(tau1*(1-tau2));
  pi_0 = 1/(1+sqrt(l_0));
  pv_s = pv_sup(supf,k,l_0);
  pv_e = pv_exp(expf,k,l_0);
  pv_a = pv_ave(avef,k,l_0);

  x1 = x[1:im,.];
  x2 = x[im+1:n,.];
  y1 = y[1:im];
  y2 = y[im+1:n];
  xx1 = invpd(moment(x1,0));
  xx2 = invpd(moment(x2,0));
  beta1 = xx1*(x1'y1);
  beta2 = xx2*(x2'y2);
  e1 = y1-x1*beta1;
  e2 = y2-x2*beta2;
  sig1 = (e1'e1)/(im-k);
  sig2 = (e2'e2)/(n-im-k);
  se1  = sqrt(diag(xx1)*sig1);
  se2  = sqrt(diag(xx2)*sig2);
  em = e1|e2;
  yd1 = y1-meanc(y1);
  yd2 = y2-meanc(y2);
  r21 = 1 - (e1'e1)/(yd1'yd1);
  r22 = 1 - (e2'e2)/(yd2'yd2);
  "Estimated Breakpoint (index)          = " im;
  "Percentage of Sample                  = " (im/n);
  "";
  "First Sample:";
  "Number of Observations                = " im;
  "Sample Variance                       = " sig1;
  "R-squared                             = " r21;
  "";"";
  "Estimates and Standard Errors from First Sample: ";"";
  yp=printfm(qvar[2:k+1]~beta1~se1,mas,fmt');
  "";"";
  "Second Sample:";
  "Number of Observations                = " (n-im);
  "Sample Variance                       = " sig2;
  "R-squared                             = " r22;
  "";"";
  "Estimates and Standard Errors from Second Sample: ";"";
  yp=printfm(qvar[2:k+1]~beta2~se2,mas,fmt');
  "";"";


@ Fixed Regressor Bootstrap  @

supfb = zeros(r2,r1);
expfb = zeros(r2,r1);
avefb = zeros(r2,r1);
supfh = zeros(r2,r1);
expfh = zeros(r2,r1);
avefh = zeros(r2,r1);

ri=1; do while ri<=r1;

    u   = rndn(n,r2);
    euf = u - x*xxi*(x'u);
    eef = sumc(euf.^2);
    ff  = zeros(n,r2);
    snf = x[1:n1-1,.]'euf[1:n1-1,.];

    uh  = u.*em;
    euh = uh - x*xxi*(x'uh);
    eeh = sumc(euh.^2);
    ffh = zeros(n,r2);
    snh = x[1:n1-1,.]'euh[1:n1-1,.];

    mf = moment(x[1:n1-1,.],0);
    mif = invpd(mf);
    msif = invpd(xx-mf);

    ib = n1; do while ib <= n2;
      xif  = x[ib,.]';
      ximf = xif'mif;
      mif  = mif - (ximf'ximf)/(1+ximf*xif);
      ximf = xif'msif;
      msif = msif + (ximf'ximf)/(1-ximf*xif);
      snf  = snf + xif*euf[ib,.];
      qf   = sumc(snf.*(msif*xx*mif*snf));
      ff[ib,.] = (qf./(eef-qf))';
      snh  = snh + xif*euh[ib,.];
      qf   = sumc(snh.*(msif*xx*mif*snh));
      ffh[ib,.] = (qf./(eeh-qf))';
    ib = ib+1; endo;

    ftf = ff[n1:n2,.]*(n-k*2);
    supfb[.,ri] = (maxc(ftf) .> supf);
    expfb[.,ri] = (meanc(exp(ftf/2)) .> exp(expf));
    avefb[.,ri] = (meanc(ftf) .> avef);

    ftf = ffh[n1:n2,.]*(n-k*2);
    supfh[.,ri] = (maxc(ftf) .> supf);
    expfh[.,ri] = (meanc(exp(ftf/2)) .> exp(expf));
    avefh[.,ri] = (meanc(ftf) .> avef);

ri=ri+1;endo;

supfb = meanc(meanc(supfb));
expfb = meanc(meanc(expfb));
avefb = meanc(meanc(avefb));

supfh = meanc(meanc(supfh));
expfh = meanc(meanc(expfh));
avefh = meanc(meanc(avefh));

"Tests for Structural Change ";
"";
"Bootstrap Replications  = " r1*r2;
"";
"              Test        Andrews      Bootstrap    Hetero-Corrected";
"              Statistic   P-Value      P-Value      P-Value";
"";"";
"SupF    " supf~pv_s~supfb~supfh;
"ExpF    " expf~pv_e~expfb~expfh;
"AveF    " avef~pv_a~avefb~avefh;
"";"";


endp;
