##  This R source file computes the Mallows Model Average (MMA) and 
##  the Jackknife Model Average (JMA) least-squares estimates.
##
##  written by
##  Bruce E. Hansen
##  Department of Economics
##  University of Wisconsin
##
##  Format:
##    
##  library(foreign)
##
##  library(quadprog) 
##
##  xy <- read.dta("Z:\\data_filename.dta")
##
##  y <- xy[,1]
##
##  x <- xy[,2:ncol(xy)]
##
##  method <- 2
##
##  subset <- 2
##
##
##
##  Notes:
##
##  (1) Make sure to change the file names and file paths of the data set and output files.
##      The outputs are saved in betahat.dat and weight.dat.
##  (2) The default value of the method is Jackknife model average estimates and the default 
##      value of the subset is all combinations of subsets.
##      method: set to 1 for Mallows model average estimates
##              set to 2 for Jackknife model average estimates          
##      subset: set to 1 for pure nested subsets
##              set to 2 for all combinations of subsets
##  (3) Package "quadprog" is required. Run the following command in R first: 
##      install.packages("quadprog")
##      Make sure the path of the package "quadprog" is correct. 
##      The user can change the default value using the following command:
##      library(quadprog,lib.loc ='U:\\') 
##  (4) The "quadprog" procedure requires the matrix of squared residuals is positive 
##      definite, otherwise the program will crash.  
##  (5) For pure nested subsets, the regressors columns should be ordered, with the 
##      intercept first and then in order of relevance. 
##  (6) For all combinations of subsets, method <- 2, the number of regressors is limited 
##      to less than 20. 
##
###############################################################################################

library(foreign)

library(quadprog)

#library(quadprog,lib.loc ='U:\\') 

xy <- read.dta("Z:\\data_filename.dta")

y <- xy[,1]

x <- xy[,2:ncol(xy)]

method <- 2

subset <- 2


gma <- function(y,x,method,subset){  
  y <- as.matrix(y)
  x <- as.matrix(x)  
  s <- as.matrix(subset)
  n <- nrow(x)
  p <- ncol(x)  
  if ((nrow(s)==1) && (ncol(s)==1)){
     if (subset == 1){
        s <- matrix(1,nrow=p,ncol=p)
        s[upper.tri(s)] <- 0
        zero <- matrix(0,nrow=1,ncol=p)
        s <- rbind(zero,s) 
     } 
     if (subset == 2){
        s <- matrix(0,nrow=2^p,ncol=p)
        s0 <- matrix(c(1,rep(0,p-1)),1,p)
        s1 <- matrix(c(rep(0,p)),1,p)
        for (i in 2:2^p){
           s1 <- s0 + s1
           for (j in 1:p){
              if (s1[1,j] == 2){
                 s1[1,j+1] <- s1[1,j+1]+1
                 s1[1,j] <- 0
              }
           }           
           s[i,] <- s1
        }   
     }   
  }   
  m <- nrow(s)
  bbeta <- matrix(0,nrow=p,ncol=m)
  if (method == 2) ee <- matrix(0,nrow=n,ncol=m)
  for (j in 1:m){
     ss <- matrix(1,nrow=n,ncol=1) %*% s[j,]
     indx1 <- which(ss[,]==1)
     xs <- as.matrix(x[indx1])
     xs <- matrix(xs,nrow=n,ncol=nrow(xs)/n)
     if (sum(ss)==0){
        xs <- x
        betas <- matrix(0,nrow=p,ncol=1)
        indx2 <- matrix(c(1:p),nrow=p,ncol=1)  
     }  
     if (sum(ss)>0){
        betas <- solve(t(xs)%*%xs)%*%t(xs)%*%y 
        indx2 <- as.matrix(which(s[j,]==1))  
     }
     beta0 <- matrix(0,nrow=p,ncol=1)
     beta0[indx2] <- betas     
     bbeta[,j] <- beta0    
     if (method == 2){
        ei <- y - xs %*% betas
        hi <- diag(xs %*% solve(t(xs) %*% xs) %*% t(xs))
        ee[,j] <- ei*(1/(1-hi))
     }
  }
  if (method == 1){
     ee <- y %*% matrix(1,nrow=1,ncol=m) - x %*% bbeta
     ehat <- y - x %*% bbeta[,m]
     sighat <- (t(ehat) %*% ehat)/(n-p)
  }
  a1 <- t(ee) %*% ee
  if (method == 1) a2 <- matrix(c(-sighat*rowSums(s)),m,1)  
  if (method == 2) a2 <- matrix(0,nrow=m,ncol=1)
  a3 <- t(rbind(matrix(1,nrow=1,ncol=m),diag(m),-diag(m)))
  a4 <- rbind(1,matrix(0,nrow=m,ncol=1),matrix(-1,nrow=m,ncol=1))
  w0 <- matrix(1,nrow=m,ncol=1)/m 
  QP <- solve.QP(a1,a2,a3,a4,1)
  w <- QP$solution
  w <- as.matrix(w)
  w <- w*(w>0)
  w <- w/sum(w0)
  betahat <- bbeta %*% w
  ybar <- mean(y)
  yhat <- x %*% betahat
  ehat <- y-yhat
  r2 <- sum((yhat-ybar)^2)/sum((y-ybar)^2)
  if (method == 1) cn=(t(w) %*% a1 %*% w - 2*t(a2) %*% w)/n
  if (method == 2) cn=(t(w) %*% a1 %*% w)/n
  list(betahat=betahat,weight=w,yhat=yhat,ehat=ehat,r2=r2,cn=cn)
}


result <- gma(y,x,method,subset)
betahat <- result$betahat
weight <- result$weight
betahat
weight
write.table(betahat,file = "Z:\\betahat.dat",quote = FALSE,row.names = FALSE,col.names = FALSE)
write.table(weight,file = "Z:\\weight.dat",quote = FALSE,row.names = FALSE,col.names = FALSE)






