/* MMACUMK.PRG 

This Gauss procedure computes the Mallows Model Average (MMA) least-squares estimates,
as described in the paper "The Asymptotic Risk of the Least Squares Averaging estimator".

written by
Chu-An Liu and Bruce E. Hansen
Department of Economics
University of Wisconsin

Format:

{betahat,w,yhat,ehat,r2,cn,wstar}=mmacumk(y,x,k);

Inputs:
y           nx1      dependent variable
x           nxp      regressor matrix
k           mx1      number of regressors for submodels 

Outputs:
betahat     px1      parameter estimate
w           mx1      weight vector
yhat        nx1      fitted values   
ehat        nx1      fitted residuals   
r2          1x1      R-squared
cn          1x1      value of Mallows criterion
wstar       (m-1)x1  cummulative weight vector

Note:
The regressors columns should be ordered, with the intercept first and then in order of relevance. 
*/

proc (7) = mmacumk(y,x,k);
local j,n,p,m,xx,sxy,kc,bbeta,ee,ehat,sighat,ff,kk,wbar,n1,n2,n3,indx,indx0,indx1,indx2,kbar,fbar,
      wstar,w,ww,betahat,ybar,yhat,r2,a1,a2,cn; 
  n=rows(x);
  p=cols(x);
  m=rows(k);
  xx=x'x;
  sxy=x'y;
  bbeta=zeros(p,m);
  kc=cumsumc(k);
  for j (1,m,1);
      if kc[j]==0; bbeta[.,j]=zeros(p,1);      
      else; bbeta[1:kc[j],j]=sxy[1:kc[j]]/xx[1:kc[j],1:kc[j]];
      endif;  
  endfor;
  ee=y-x*bbeta;
  ehat=y-x*bbeta[.,m];
  sighat=(ehat'ehat)/(n-p);
  ff=zeros(m-1,1);
  for j (1,m-1,1);  
      ff[j]=(ee[.,j]'ee[.,j]-ee[.,j+1]'ee[.,j+1])/sighat;
  endfor;
  kk=k[2:m,1];
  wbar=kk./ff;
  n1=rows(wbar);
  indx=wbar[1:n1-1].>=wbar[2:n1];
  indx0=seqa(1,1,m-1);  
  indx1=0|indx;
  do while sumc(indx)>0;
      kbar=kk;
      fbar=ff;
      for j (1,m-1,1);
          if indx1[j]==1;
              kbar[j]=kbar[j-1]+kbar[j];
              fbar[j]=fbar[j-1]+fbar[j];
              kbar[j-1]=0;
              fbar[j-1]=0;          
          endif; 
      endfor;
      kbar=delif(kbar,kbar.==0);
      fbar=delif(fbar,fbar.==0);
      wbar=kbar./fbar;
      n2=rows(wbar);
      if n2>1;
          indx=wbar[1:n2-1].>=wbar[2:n2];
          indx2=0|indx;
      else;
          indx=0;
          indx2=0;
      endif;
      indx1[selif(indx0,indx1.==0)]=indx2;
  endo;
  wbar=wbar.*(wbar.<1)+1*(wbar.>=1);
  n3=rows(wbar); 
  wstar=zeros(m-1,1);
  wstar[selif(indx0,indx1.==0)]=wbar;
  w=(indx1.==0);
  if n3>1;
      ww=wbar[1]|(wbar[2:n3]-wbar[1:n3-1]);
  else;
      ww=wbar[1];
  endif; 
  w[selif(indx0,w.>0)]=ww;
  w=w|(1-wbar[n3]);
  betahat=bbeta*w;
  ybar=meanc(y);
  yhat=x*betahat;
  ehat=y-yhat;
  r2=sumc((yhat-ybar)^2)/sumc((y-ybar)^2);
  a1=ee'ee;
  a2=kc*sighat;
  cn=(w'a1*w+2*a2'w)/n;
retp(betahat,w,yhat,ehat,r2,cn,wstar);
endp;