function [betahat,w,yhat,ehat,r2,cn] = gma(y,x,method,subset,s)

%  This Matlab function computes the Mallows Model Average (MMA) and 
%  the Jackknife Model Average (JMA) least-squares estimates.
%
%  written by
%  Chu-An Liu and Bruce E. Hansen
%  Department of Economics
%  University of Wisconsin
%
%  Format: 
%
%  [betahat,w,yhat,ehat,R2,Cn]=gma(y,x,method,subset,s)
%  or  [betahat,w,yhat,ehat,R2,Cn]=gma(y,x,method,subset)
%  
%  Inputs:
%  y           nx1   dependent variable
%  x           nxp   regressor matrix
%  method      1x1   set to 1 for Mallows model average estimates
%                    set to 2 for Jackknife model average estimates          
%  subset      1x1   set to 1 for pure nested subsets
%                    set to 2 for all combinations of subsets
%                    set to 3 for using the selection matrix
%  s           mxp   selection matrix. optional. m: number of models 
%                    Example: 
%                       Suppose there are 3 candidate models.
%                       Model 1: y=beta1*x1+beta2*x2+e
%                       Model 2: y=beta1*x1+beta3*x3+e
%                       Model 3: y=beta1*x1+beta2*x2+beta4*x4+e
%                       Then s=[1,1,0,0;
%                               1,0,1,0;
%                               1,1,0,1]
%
%  Outputs:
%  betahat     px1   parameter estimate
%  w           mx1   weight vector
%  yhat        nx1   fitted values   
%  ehat        nx1   fitted residuals   
%  r2          1x1   R-squared
%  cn          1x1   Value of Mallows criterion or Cross-Validation criterion
%
%  Note:
%  For pure nested subsets, the regressors columns should be ordered, with the 
%  intercept first and then in order of relevance. 
%  For all combinations of subsets, p is less than about 20. 
% 

[n,p]=size(x);
if nargin<5;
    if nargin<4; subset=1; 
        if nargin<3; method=1;
        end
    end
    if subset==1;
        s=tril(ones(p+1,p+1),-1);
        s=s(:,1:end-1);
    elseif subset==2;
        s=zeros(2^p,p);
        s0=[1,zeros(1,p-1)];
        s1=zeros(1,p);
        for i=2:2^p;
            s1=s0+s1;
            for j=1:p;
                if s1(j)==2;
                    s1(j+1)=s1(j+1)+1;
                    s1(j)=0;
                end
            end
            s(i,:)=s1;
        end
    end          
end
if nargin<2;
     error('Wrong number of input arguments')
end
m=length(s(:,1));
bbeta=zeros(p,m);
if method==2; ee=zeros(n,m); end
for j=1:m;
    ss=ones(n,1)*s(j,:)>0;
    xs=x(ss);
    xs=reshape(xs,n,length(xs)/n);
    betas=(xs'*xs)\(xs'*y);
    sj=s(j,:)>0;
    bbeta(sj,j)=betas;
    if method==2;
        ei=y-xs*betas;
        hi=diag(xs/(xs'*xs)*xs');
        ee(:,j)=ei.*(1./(1-hi));
    end
end
if method==1;
    ee=y*ones(1,m)-x*bbeta;
    ehat=y-x*bbeta(:,m);
    sighat=(ehat'*ehat)/(n-p);
end
a1=ee'*ee;
if method==1; a2=sum(s,2)*sighat;
elseif method==2; a2=zeros(m,1);
end
w0=ones(m,1)/m;
options=optimset('LargeScale','off','Display','off');
w=quadprog(a1,a2,zeros(1,m),0,ones(1,m),1,zeros(m,1),ones(m,1),w0,options);
w=w.*(w>0);
w=w/sum(w);
betahat=bbeta*w;
ybar=mean(y);
yhat=x*betahat;
ehat=y-yhat;
r2=sum((yhat-ybar).^2)/sum((y-ybar).^2);
if method==1; cn=(w'*a1*w+2*a2'*w)/n;
elseif method==2; cn=w'*a1*w/n;
end




