##  This R function computes the Mallows Model Average (MMA) and 
##  the Jackknife Model Average (JMA) least-squares estimates.
##
##  written by
##  Bruce E. Hansen
##  Department of Economics
##  University of Wisconsin
##
##  Format:
## 
##  result <- gma(y,x,method,subset)
##  result$betahat
##  
##  Inputs:
##  y           nx1   dependent variable
##  x           nxp   regressor matrix
##  method      1x1   set to 1 for Mallows model average estimates
##                    set to 2 for Jackknife model average estimates          
##  subset      1x1   set to 1 for pure nested subsets
##                    set to 2 for all combinations of subsets
##              mxp   input the (mxp) selection matrix, where m is 
##                    the number of models 
##                    Example: 
##                       Suppose there are 3 candidate models.
##                       Model 1: y=beta1*x1+beta2*x2+e
##                       Model 2: y=beta1*x1+beta3*x3+e
##                       Model 3: y=beta1*x1+beta2*x2+beta4*x4+e
##                       Then subset <- matrix(c(1,1,1,1,0,1,0,1,0,0,0,1),3,4)
##
##  Outputs:
##  betahat     px1   parameter estimate
##  w           mx1   weight vector
##  yhat        nx1   fitted values   
##  ehat        nx1   fitted residuals   
##  r2          1x1   R-squared
##  cn          1x1   Value of Mallows criterion or Cross-Validation criterion
##  
##  Note:
##  Package "quadprog" is required.
##  For pure nested subsets, the regressors columns should be ordered, with the 
##  intercept first and then in order of relevance. 
##  For all combinations of subsets, p is less than about 20. 

gmaN <- function(y,x,method,subset){ 
 
  y <- as.matrix(y)
  x <- as.matrix(x)  
  s <- as.matrix(subset)
  n <- nrow(x)
  p <- ncol(x)
  
  if ((nrow(s)==1) && (ncol(s)==1)){
     if (subset == 1){
        s <- matrix(1,nrow=p,ncol=p)
        s[upper.tri(s)] <- 0
        zero <- matrix(0,nrow=1,ncol=p)
        s <- rbind(zero,s) 
     } 
     if (subset == 2){
        s <- matrix(0,nrow=2^p,ncol=p)
        s0 <- matrix(c(1,rep(0,p-1)),1,p)
        s1 <- matrix(c(rep(0,p)),1,p)
        for (i in 2:2^p){
           s1 <- s0 + s1
           for (j in 1:p){
              if (s1[1,j] == 2){
                 s1[1,j+1] <- s1[1,j+1]+1
                 s1[1,j] <- 0
              }
           }           
           s[i,] <- s1
        }   
     }   
  }   

  m <- nrow(s)
  bbeta <- matrix(0,nrow=p,ncol=m)
  if (method == 2) ee <- matrix(0,nrow=n,ncol=m)

  for (j in 1:m){
     ss <- matrix(1,nrow=n,ncol=1) %*% s[j,]
     indx1 <- which(ss[,]==1)
     xs <- as.matrix(x[indx1])
     xs <- matrix(xs,nrow=n,ncol=nrow(xs)/n)
     if (sum(ss)==0){
        xs <- x
        betas <- matrix(0,nrow=p,ncol=1)
        indx2 <- matrix(c(1:p),nrow=p,ncol=1)  
     }  
     if (sum(ss)>0){
        betas <- solve(t(xs)%*%xs)%*%t(xs)%*%y 
        indx2 <- as.matrix(which(s[j,]==1))  
     }
     beta0 <- matrix(0,nrow=p,ncol=1)
     beta0[indx2] <- betas     
     bbeta[,j] <- beta0    
     if (method == 2){
        ei <- y - xs %*% betas
        hi <- diag(xs %*% solve(t(xs) %*% xs) %*% t(xs))
        ee[,j] <- ei*(1/(1-hi))
     }
  }

  if (method == 1){
     ee <- y %*% matrix(1,nrow=1,ncol=m) - x %*% bbeta
     ehat <- y - x %*% bbeta[,m]
     sighat <- (t(ehat) %*% ehat)/(n-p)
  }
  
  a1 <- t(ee) %*% ee
  if (qr(a1)$rank<ncol(ee)) a1 <- a1 + diag(m)*1e-10
  if (method == 1) a2 <- matrix(c(-sighat*rowSums(s)),m,1)  
  if (method == 2) a2 <- matrix(0,nrow=m,ncol=1)
  a3 <- t(rbind(matrix(1,nrow=1,ncol=m),diag(m),-diag(m)))
  a4 <- rbind(1,matrix(0,nrow=m,ncol=1),matrix(-1,nrow=m,ncol=1))

  w0 <- matrix(1,nrow=m,ncol=1)/m 
  QP <- solve.QP(a1,a2,a3,a4,1)
  w <- QP$solution
  w <- as.matrix(w)
  w <- w*(w>0)
  w <- w/sum(w0)
  betahat <- bbeta %*% w
  ybar <- mean(y)
  yhat <- x %*% betahat
  ehat <- y-yhat
  r2 <- sum((yhat-ybar)^2)/sum((y-ybar)^2)
  if (method == 1) cn=(t(w) %*% a1 %*% w - 2*t(a2) %*% w)/n
  if (method == 2) cn=(t(w) %*% a1 %*% w)/n
  list(betahat=betahat,w=w,yhat=yhat,ehat=ehat,r2=r2,cn=cn)
}


