/* IR.PRG 

This Gauss program calculates asymptotic confidence intervals
for an impulse response function from an AR(p)
 using the no-rejection region from a GMM criterion function. 

The model is a linear autoregression in Y with p lags.

The user must provide:
  -  The data in a vector "dat"
  -  p (number of AR lags)
  -  kmax (maximum number of impulse responses).
  - The confidence level
  -  Number of GMM iterations (set to zero for evaluation of weight matrix using OLS estimates)
*/

/****** Data in "dat" vector *****/

load dat[]=gdpq.dat;
dat=log(dat);
dat=dat[2:rows(dat)]-dat[1:rows(dat)-1];
n=rows(dat);
dat=dat[2:n]-dat[1:n-1];

/******** Controls *********/

p = 2;		@ number of AR lags @
kmax = 12	; 	@ number of impulse responses @
_level = .90; 	@ Level of confidence region @
iter = 0;		@ GMM iterations @


/****************************************************/

coset;
string _co_Options = {stepbt newton forward none};
			@  This can be changed. In particular, the choice
			      of step length method may make a difference @
_co_MaxIters=500;    		@  This can be changed if desired @
trap 4;
_co_EqProc=&g_contraint_theta;
_co_EqJacobian=&g_derivative;
_co_GradProc=&qmm_grad;
_co_HessProc=&qmm_hess;

/******************************************/

n=rows(dat);
y=dat[p+1:n];
x=dat[p:n-1];
for j (2,p,1);
x=x~dat[p+1-j:n-j];
endfor;
x=x~ones(n-p,1);

n=rows(y);cx=cols(x);
cr=cdfchii(_level,1);
xxi=invpd(moment(x,0));
betahat=xxi*(x'y);
e=y-x*betahat;
v=xxi*moment(x.*e,0)*xxi;

/*********************************/

ir_est=zeros(kmax,1);
ir_se=zeros(kmax,1);
ir_ci=zeros(kmax,2);
ir_check=zeros(kmax,2);

"Computing Impulse Responses and Confidence Intervals"; "";
$("Horizon"~"Lower"~"Estimate"~"Upper");"";
k=1; do while k<= kmax;

  thetahat=g_function(betahat);
  h=g_derivative(betahat)';
  setheta=sqrt(h'v*h);
  ir_est[k]=thetahat;
  ir_se[k]=setheta;
  theta=thetahat;
  if k==1;
    r1=sqrt(cr)*setheta; r2=r1;
    a1=betahat; a2=betahat;
  endif;
  {a1,c1,i1}=c_solve(a1,thetahat-r1,cr);
  {a2,c2,i2}=c_solve(a2,thetahat+r2,cr);
  ir_ci[k,.]=c1~c2;
  ir_check[k,.]=i1~i2;
  r1=thetahat-c1;
  r2=c2-thetahat;
  if i1==1; a1=betahat;   r1=sqrt(cr)*setheta; endif;
  if i2==1; a2=betahat;   r2=sqrt(cr)*setheta; endif;

k~c1~thetahat~c2;
k=k+1;endo;
"";
/*********************************/

lev=ftocv(_level*100,1,0);
tit1="" $+ "Impulse Response and GMM Asymptotic " $+ lev $+ "% Confidence Intervals";
ks=seqa(1,1,kmax);

graphset;
_plwidth=2;
_pdate="";
let _pltype=6 4 4;
let _pcolor = 14 15 15;
_pcolor=15;
title(tit1);
xlabel("Horizon");
ylabel("Impulse Response");
xy(ks,ir_est~ir_ci);


/***********************************/

/**** Proc for Theta=g(alpha) ********/
proc g_function(a);
local p,aa,aaa,j,c,am,ak;
  p=rows(a)-1;
  if p==1; c=a^k; 
  elseif k==1; c=a[1];
  elseif k==2; c=(a[1]^2)+a[2];
  else;
    if k<p; 
      ak=a[1:k];
      aa=(ak')|(eye(k-1)~zeros(k-1,1));
    else;
      ak=a[1:p];
      aa=(ak')|(eye(p-1)~zeros(p-1,1));
    endif;
    am=ak'aa;
    for j (4,k,1);
      am=am*aa;
    endfor;
    c=am*aa[.,1];
  else;
  endif;
retp(c);
endp;


/**** Proc for d/da g(a) ********/

proc g_derivative(a);
local p,d,phi,ak,aa,am,j,st,a1;
  p=rows(a)-1; d=zeros(1,p); 
  if p==1; d=k*(a[1]^(k-1));
  elseif k==1; d[1]=1;
  elseif k==2; d[1]=2*a[1];d[2]=1;
  elseif (k==3); d[1]=3*(a[1]^2)+2*a[2];d[2]=2*a[1];  if p>2; d[3]=1;endif;
  else;
    d=1|a[1]|((a[1]^2)+a[2])|zeros(k-3,1);
    ak=a[1:p];
    aa=(ak')|(eye(p-1)~zeros(p-1,1));
    am=ak';
    a1=aa[.,1];
    for j (4,k,1);
      am=am*aa;
      d[j]=am*a1;
    endfor;
    st=zeros(p,1);
    d=recserar(st|d,st,ak);
    d=rev(d[p+k-p+1:p+k])';
  endif;
  d=d~0;
retp(d);
endp;  


proc g_contraint_theta(b);
local thetav;
 thetav=varget("theta");
retp(g_function(b)-thetav);
endp;

proc qmm(b);
local xe,xv,yv,omiv;
  xv=varget("x");
  yv=varget("y");
  xe=xv'yv-moment(xv,0)*b;
  omiv=varget("omi");
retp(xe'omiv*xe);
endp;

proc qmm_grad(b);
local xv,yv,xx,omiv,xe;
  xv=varget("x");
  yv=varget("y");
  xx=moment(xv,0);
  xe=xv'yv-xx*b;
  omiv=varget("omi");
retp(-xx*omiv*xe*2);
endp;

proc qmm_hess(b);
  local xv,omiv,xx;
  xv=varget("x");
  xx=moment(xv,0);
  omiv=varget("omi");
retp(xx*omiv*xx*2);
endp;


proc (3) = profile_gmm(thetav,beta0,iter);
local theta_ok,omiv,omi_ok,beta1,g,gr,ret,i,xv,yv;
  xv=varget("x");
  yv=varget("y");
  theta_ok=varput(thetav,"theta");
  omiv=invpd(moment(xv.*(yv-xv*(yv/xv)),0));
  omi_ok=varput(omiv,"omi");
  {beta1,g,gr,ret}=co(&qmm,beta0);
  i=1; do while i<=iter;
    omiv=invpd(moment(xv.*(yv-xv*beta1),0));
    omi_ok=varput(omiv,"omi");
    {beta1,g,gr,ret}=co(&qmm,beta1);
  i=i+1;endo;
 retp(g,beta1,ret);
endp;


@ 
The following procedure finds the endpoints of the confidence interval
cr is the critical value
a0 is a starting value for the parameter vector
c0 is an initial guess of the endpoint
set c0>thetahat to find upper endpoint 
set c0<thetahat to find lower endpoint 
@
proc (3) = c_solve(a0,c0,cr);
local gg,g_r,g_l,dtheta,theta_r,theta_l,a1,g,gr,ret,i,j,w,nogo,try;
  theta=varget("thetahat");
  gg=0; g_l=0;  theta_l=theta; g_r=0;  theta_r=theta;
  dtheta=(c0-thetahat)*.6;  
  nogo=1;
  for j (1,100,1);
    theta=theta+dtheta;
    {g,a1,ret}=profile_gmm(theta,a0,iter);
    if ret>2;
       theta=theta-dtheta;
       dtheta=dtheta/2;
    else;
      gg=g; 
      theta_l=theta_r;
      g_l=g_r;
      theta_r=theta;
      g_r=gg;
    endif;
    try = (abs(cr-gg) .< .01) +(gg>cr);
    if try==1;
      nogo=0; break;
    endif;
  endfor;
  if nogo==1; goto _end; endif;
  try= (abs(cr-g_r) .< .01)+(abs(theta_l-theta_r) .< .000001);
  if try==1; goto _end; endif;
  a1=a0; nogo=1;
  for j (1,20,1);
    theta=(theta_l+theta_r)/2;
    {g,a1,ret}=profile_gmm(theta,a1,iter);
    if ret>2;
      {g,a1,ret}=profile_gmm(theta,a0,iter);
    endif;
    if g>cr;
      theta_r=theta; g_r=g;
    else;
      theta_l=theta; g_l=g;
    endif;
    try= (abs(cr-g) .< .01)+(abs(theta_l-theta_r) .< .000001);
    if try==1; nogo=0; break; endif;
    theta=(theta_l*(g_r-cr)+theta_r*(cr-g_l))/(g_r-g_l);
    {g,a1,ret}=profile_gmm(theta,a1,iter);
    if ret>2;
      {g,a1,ret}=profile_gmm(theta,a0,iter);
    endif;
    if g>cr;
      theta_r=theta; g_r=g;
    else;
      theta_l=theta; g_l=g;
    endif;
    try= (abs(cr-g) .< .01)+(abs(theta_l-theta_r) .< .000001);
    if try==1; nogo=0; break; endif;
  endfor;
  _end:
retp(a1,theta,nogo);
endp;



/***********************************/
