/* GMM_IR.PRG 

This Gauss program calculates asymptotic confidence intervals
for an impulse response from an AR(p)
using the no-rejection region from a GMM criterion function. 

The model is a linear autoregression in Y with p lags.

The user must provide:
  -  The data in a vector "dat"
  -  p (number of AR lags)
  -  k (impulse response horizon).
  -  The confidence level

The output is a graph of the profile GMM criterion for the impulse response

There are also a number of optional controls, allowing for
  -  Number of gridpoints for graph
  -  Name for the output file
  -  Number of GMM iterations (set to zero for evaluation of weight matrix using OLS estimates)

The procedure requires the GAUSS module "Constrained Optimization" and "pgraph".
Before running the procedure, make sure the CO library is in memory.
To do this, you can type at the Gauss prompt: "library co,pgraph".



*/

/****** Data in "dat" vector *****/

load dat[]=gdpq.dat;
dat=log(dat);
dat=dat[2:rows(dat)]-dat[1:rows(dat)-1];



/****************** Controls ************************/

p = 2;		@ number of AR lags @
k = 4; 		@ impulse response @

_level = .90;	@ Level of confidence region @
_grd = 101;	@ Number of gridpoints for graph of GMM criterion function @
iter = 1;		@ GMM iterations @
output file = "gmm_ir.out" reset; output off;

/****************************************************/

coset;
string _co_Options = {stepbt newton forward none};
			@  This can be changed. In particular, the choice
			      of step length method may make a difference @
_co_MaxIters=500;    		@  This can be changed if desired @
trap 4;
_co_EqProc=&g_contraint_theta;
_co_EqJacobian=&g_derivative;
_co_GradProc=&qmm_grad;
_co_HessProc=&qmm_hess;

/******************************************/

n=rows(dat);
y=dat[p+1:n];
x=dat[p:n-1];
for j (2,p,1);
x=x~dat[p+1-j:n-j];
endfor;
x=x~ones(n-p,1);

n=rows(y);cx=cols(x);
cr=cdfchii(_level,1);
xxi=invpd(moment(x,0));
betahat=xxi*(x'y);
e=y-x*betahat;
v=xxi*moment(x.*e,0)*xxi;
se=sqrt(diag(v));
thetahat=g_function(betahat);
h=g_derivative(betahat)';
setheta=sqrt(h'v*h);
_names=0 $+ "AR(" $+ ftocv(seqa(1,1,cx-1),1,0) $+ ")";
_names=_names|"Interct";
output on;
"OLS Estimates of Linear Model";
"";
pr=printfmt("Variable"~"Estimate"~"St Error",0~0~0);"";
pr=printfmt(_names~betahat~se,0~1~1);"";
pr=printfmt("Theta"~thetahat~setheta,0~1~1);"";
"";
output off;
"";"";"";
/*********************************/

lev=ftocv(_level*100,1,0);
kk=ftocv(k,1,0);
tit1="" $+ lev $+ "% GMM Confidence Interval for " $+ kk $+ "th Impulse Response";
tit2="" $+ "GMM Criterion and " $+ lev $+ "% Critical Value";
if iter==0;
  tit1=tit1 $+ "\LGMM Weight Matrix Calculated by OLS";
elseif iter==1;
  tit1=tit1 $+ "\LGMM Weight Matrix Calculated with 1 Iteration";
else;
  tit1=tit1 $+ "\LGMM Weight Matrix Calculated with "
$+ ftocv(iter,1,0) $+ " Iterations";
endif;

  "Computing GMM on grid for Graph";
  a0=betahat;
  gm=0;j=0;
  do while gm < (cr+1); 
    j=j+1;
    theta=thetahat-setheta*j;
    {gm,a0,ret}=profile_gmm(theta,a0,iter);
    theta~gm~ret;
  endo;
  "";
  theta1=theta;
  a0=betahat;
  gm=0;j=0;
  do while gm < (cr+1); 
    j=j+1;
    theta=thetahat+setheta*j;
    {gm,a0,ret}=profile_gmm(theta,a0,iter);
    theta~gm~ret;
  endo;
  "";
  theta2=theta;
  aa=seqa(theta1,(theta2-theta1)/(_grd-1),_grd);
  gms=zeros(_grd,1);
  for i (_grd,1,-1);
    theta=aa[i];
    {gms[i],a0,ret}=profile_gmm(theta,a0,iter);
    theta~gms[i]~ret;
  endfor;
  "";"";
  z=ones(_grd,1)*cr;
  d=(gms.<cr);
  d1=maxindc(d); 
  d2=_grd+1-maxindc(rev(d));
  c1a=aa[d1-1];
  c1b=aa[d1];
  c1=((gms[d1-1]-cr)*c1b+(cr-gms[d1])*c1a)/(gms[d1-1]-gms[d1]);
  c2a=aa[d2];
  c2b=aa[d2+1];
  c2=((gms[d2+1]-cr)*c2a+(cr-gms[d2])*c2b)/(gms[d2+1]-gms[d2]);

  graphset;
  pqgwin many;
  _pdate="";
  _plwidth=8;
  let _pmcolor = 0 0 0 0 0 0 0 0 15;
  let _pcolor = 1 2 9 12;
  fonts("complex");
  title(tit1);
  xtit="" $+ kk $+ "th Impulse Response";
  xlabel(xtit);
  ylabel(tit2);
  _plwidth=2;
  _pdate="";
  _psym=
  (c1~cr~2~3~15~1~0)|
  (c2~cr~2~3~15~1~0);
  xy(aa,gms~z);
output on;
$tit1;
c1~c2;
"";"";
tit = "" $+ "Computed on Grid with " $+ ftocv(_grd,1,0) $+ " Gridpoints";
$tit;
"GMM Iterations " iter;
"";
output off;

/***********************************/

/**** Proc for Theta=g(alpha) ********/
proc g_function(a);
local p,aa,aaa,j,c,am,ak;
  p=rows(a)-1;
  if p==1; c=a^k; 
  elseif k==1; c=a[1];
  elseif k==2; c=(a[1]^2)+a[2];
  else;
    if k<p; 
      ak=a[1:k];
      aa=(ak')|(eye(k-1)~zeros(k-1,1));
    else;
      ak=a[1:p];
      aa=(ak')|(eye(p-1)~zeros(p-1,1));
    endif;
    am=ak'aa;
    for j (4,k,1);
      am=am*aa;
    endfor;
    c=am*aa[.,1];
  else;
  endif;
retp(c);
endp;


/**** Proc for d/da g(a) ********/

proc g_derivative(a);
local p,d,phi,ak,aa,am,j,st,a1;
  p=rows(a)-1; d=zeros(1,p); 
  if p==1; d=k*(a[1]^(k-1));
  elseif k==1; d[1]=1;
  elseif k==2; d[1]=2*a[1];d[2]=1;
  elseif (k==3); d[1]=3*(a[1]^2)+2*a[2];d[2]=2*a[1];  if p>2; d[3]=1;endif;
  else;
    d=1|a[1]|((a[1]^2)+a[2])|zeros(k-3,1);
    ak=a[1:p];
    aa=(ak')|(eye(p-1)~zeros(p-1,1));
    am=ak';
    a1=aa[.,1];
    for j (4,k,1);
      am=am*aa;
      d[j]=am*a1;
    endfor;
    st=zeros(p,1);
    d=recserar(st|d,st,ak);
    d=rev(d[p+k-p+1:p+k])';
  endif;
  d=d~0;
retp(d);
endp;  


proc g_contraint_theta(b);
local thetav;
 thetav=varget("theta");
retp(g_function(b)-thetav);
endp;

proc qmm(b);
local xe,xv,yv,omiv;
  xv=varget("x");
  yv=varget("y");
  xe=xv'yv-moment(xv,0)*b;
  omiv=varget("omi");
retp(xe'omiv*xe);
endp;

proc qmm_grad(b);
local xv,yv,xx,omiv,xe;
  xv=varget("x");
  yv=varget("y");
  xx=moment(xv,0);
  xe=xv'yv-xx*b;
  omiv=varget("omi");
retp(-xx*omiv*xe*2);
endp;

proc qmm_hess(b);
  local xv,omiv,xx;
  xv=varget("x");
  xx=moment(xv,0);
  omiv=varget("omi");
retp(xx*omiv*xx*2);
endp;


proc (3) = profile_gmm(thetav,beta0,iter);
local theta_ok,omiv,omi_ok,beta1,g,gr,ret,i,xv,yv;
  xv=varget("x");
  yv=varget("y");
  theta_ok=varput(thetav,"theta");
  omiv=invpd(moment(xv.*(yv-xv*(yv/xv)),0));
  omi_ok=varput(omiv,"omi");
  {beta1,g,gr,ret}=co(&qmm,beta0);
  i=1; do while i<=iter;
    omiv=invpd(moment(xv.*(yv-xv*beta1),0));
    omi_ok=varput(omiv,"omi");
    {beta1,g,gr,ret}=co(&qmm,beta1);
  i=i+1;endo;
 retp(g,beta1,ret);
endp;


/***********************************/
