/* GMM_CI.PRG 

This Gauss program calculates asymptotic confidence intervals
using the no-rejection region from a GMM criterion function. 

The model is a linear regression
Y=Xb + e
with independent observations. 

The parameter of interest, theta, is a non-linear
function of the regression parameters: theta = g(b)

The end product is a confidence interval for theta.

The user must provide:
  -  The data matrices: Y and X
  -  A procedure "g_function" to calculate theta=g(b)
  -  A procedure "g_derivative" to calculate d/db g(b)
     The procedure should return a 1xk vector.
     If analytical derivatives are unable, you can use numerical derivatives.
     (An example is given below.)
  -  The confidence level

There are also a number of optional controls, allowing for
  -  Whether or not to calculate a graph of the GMM criterion
  -  Number of gridpoints for graph (if latter calculated)
  -  Name for the output file
  -  Number of GMM iterations (set to zero for evaluation of weight matrix using OLS estimates)

The procedure requires the GAUSS module "Constrained Optimization" and "pgraph.
Before running the procedure, make sure the CO library is in memory.
To do this, you can type at the Gauss prompt: "library co,pgraph".



*/

/**** Data in Y and X matrices *****/

load dat[50,4]=gmm_ci.dat;
y=dat[.,1];
x=dat[.,2:4];

/**** Proc for Theta=g(alpha) ********/

proc g_function(b);
local t;
  t=b[1]-1/b[2];
retp(t);
endp;

/**** Proc for d/db g(b) ********/

proc g_derivative(b);
local d;
  d=1~(1/b[2]^2)~0;
retp(d);
endp;

/*  
Numerical Derivative:  Use if analytic derivatives unavailable:
proc g_derivative(b);  
local d;
  d=gradp(&g_constraint(b),b);
retp(d);
endp;
*/


/****************** Controls ************************/

_level = .95;
	@ Level of confidence region @
_graph = 1;
	@ Set to 1 to compute and display graph
	  of GMM criterion function
	  Note:  This takes extra computation time @
_grd = 101;
	@ Number of gridpoints for graph of GMM criterion function 
	  (only relevant if _graph=1)  @
iter = 0;	@ GMM iterations @
output file = "gmm_ci.out" reset; output off;

/****************************************************/

coset;
string _co_Options = {stepbt newton forward none};
			@  This can be changed. In particular, the choice
			      of step length method may make a difference @
_co_MaxIters=500;    		@  This can be changed if desired @
trap 4;
_co_EqProc=&g_contraint_theta;
_co_EqJacobian=&g_derivative;
_co_GradProc=&qmm_grad;
_co_HessProc=&qmm_hess;


/******************************************/

n=rows(y);cx=cols(x);
cr=cdfchii(_level,1);
xxi=invpd(moment(x,0));
betahat=xxi*(x'y);
e=y-x*betahat;
v=xxi*moment(x.*e,0)*xxi;
se=sqrt(diag(v));
thetahat=g_function(betahat);
h=g_derivative(betahat)';
setheta=sqrt(h'v*h);
_names=0 $+ "Alpha" $+ ftocv(seqa(1,1,cx),1,0);
output on;
"OLS Estimates of Linear Model";
"";
pr=printfmt("Variable"~"Estimate"~"St Error",0~0~0);"";
pr=printfmt(_names~betahat~se,0~1~1);"";
pr=printfmt("Theta"~thetahat~setheta,0~1~1);"";
"";
output off;
"";"";"";
/*********************************/

lev=ftocv(_level*100,1,0);
tit1="" $+ lev $+ "% GMM Asymptotic Confidence Interval for Theta";
tit2="" $+ "GMM Criterion and " $+ lev $+ "% Critical Value";
if iter==0;
  tit1=tit1 $+ "\LGMM Weight Matrix Calculated by OLS";
elseif iter==1;
  tit1=tit1 $+ "\LGMM Weight Matrix Calculated with 1 Iteration";
else;
  tit1=tit1 $+ "\LGMM Weight Matrix Calculated with "
$+ ftocv(iter,1,0) $+ " Iterations";
endif;

if _graph == 1;
  "Computing GMM on grid for Graph";
  a0=betahat;
  gm=0;j=0;
  do while gm < (cr+1); 
    j=j+1;
    theta=thetahat-setheta*j;
    {gm,a0,ret}=profile_gmm(theta,a0,iter);
    theta~gm~ret;
  endo;
  "";
  theta1=theta;
  a0=betahat;
  gm=0;j=0;
  do while gm < (cr+1); 
    j=j+1;
    theta=thetahat+setheta*j;
    {gm,a0,ret}=profile_gmm(theta,a0,iter);
    theta~gm~ret;
  endo;
  "";
  theta2=theta;
  aa=seqa(theta1,(theta2-theta1)/(_grd-1),_grd);
  gms=zeros(_grd,1);
  for i (_grd,1,-1);
    theta=aa[i];
    {gms[i],a0,ret}=profile_gmm(theta,a0,iter);
    theta~gms[i]~ret;
  endfor;
  "";"";
  z=ones(_grd,1)*cr;
  d=(gms.<cr);
  d1=maxindc(d); 
  d2=_grd+1-maxindc(rev(d));
  c1a=aa[d1-1];
  c1b=aa[d1];
  c1=((gms[d1-1]-cr)*c1b+(cr-gms[d1])*c1a)/(gms[d1-1]-gms[d1]);
  c2a=aa[d2];
  c2b=aa[d2+1];
  c2=((gms[d2+1]-cr)*c2a+(cr-gms[d2])*c2b)/(gms[d2+1]-gms[d2]);

  graphset;
  pqgwin many;
  _pdate="";
  _plwidth=8;
  let _pmcolor = 0 0 0 0 0 0 0 0 15;
  let _pcolor = 1 2 9 12;
  fonts("complex");
  title(tit1);
  xlabel("Theta");
  ylabel(tit2);
  _plwidth=2;
  _pdate="";
  _psym=
  (c1~cr~2~3~15~1~0)|
  (c2~cr~2~3~15~1~0);
  xy(aa,gms~z);
else;
  theta1=thetahat-sqrt(cr)*setheta;
  {a1,c1,i1}=c_solve(betahat,theta1,cr);
  theta2=thetahat+sqrt(cr)*setheta;
  {a2,c2,i2}=c_solve(betahat,theta2,cr);
  "";"";
endif;
output on;
$tit1;
c1~c2;
"";"";
if _graph==1;
tit = "" $+ "Computed on Grid with " $+ ftocv(_grd,1,0) $+ " Gridpoints";
$tit;
else;
"Computed by Numerical Search";
endif;
"GMM Iterations " iter;
"";
output off;

/***********************************/

proc g_contraint_theta(b);
local thetav;
 thetav=varget("theta");
retp(g_function(b)-thetav);
endp;

proc qmm(b);
local xe,xv,yv,omiv;
  xv=varget("x");
  yv=varget("y");
  xe=xv'yv-moment(xv,0)*b;
  omiv=varget("omi");
retp(xe'omiv*xe);
endp;

proc qmm_grad(b);
local xv,yv,xx,omiv,xe;
  xv=varget("x");
  yv=varget("y");
  xx=moment(xv,0);
  xe=xv'yv-xx*b;
  omiv=varget("omi");
retp(-xx*omiv*xe*2);
endp;

proc qmm_hess(b);
  local xv,omiv,xx;
  xv=varget("x");
  xx=moment(xv,0);
  omiv=varget("omi");
retp(xx*omiv*xx*2);
endp;


proc (3) = profile_gmm(thetav,beta0,iter);
local theta_ok,omiv,omi_ok,beta1,g,gr,ret,i,xv,yv;
  xv=varget("x");
  yv=varget("y");
  theta_ok=varput(thetav,"theta");
  omiv=invpd(moment(xv.*(yv-xv*(yv/xv)),0));
  omi_ok=varput(omiv,"omi");
  {beta1,g,gr,ret}=co(&qmm,beta0);
  i=1; do while i<=iter;
    omiv=invpd(moment(xv.*(yv-xv*beta1),0));
    omi_ok=varput(omiv,"omi");
    {beta1,g,gr,ret}=co(&qmm,beta1);
  i=i+1;endo;
 retp(g,beta1,ret);
endp;


@ 
The following procedure finds the endpoints of the confidence interval
cr is the critical value
a0 is a starting value for the parameter vector
c0 is an initial guess of the endpoint
set c0>thetahat to find upper endpoint 
set c0<thetahat to find lower endpoint 
@
proc (3) = c_solve(a0,c0,cr);
local gg,g_r,g_l,dtheta,theta_r,theta_l,a1,g,gr,ret,i,j,w,nogo,try;
  theta=varget("thetahat");
  gg=0; g_l=0;  theta_l=theta; g_r=0;  theta_r=theta;
  dtheta=(c0-thetahat)*.6;  
  nogo=1;
  for j (1,100,1);
    theta=theta+dtheta;
    {g,a1,ret}=profile_gmm(theta,a0,iter);
    if ret>2;
       theta=theta-dtheta;
       dtheta=dtheta/2;
    else;
      gg=g; 
      theta_l=theta_r;
      g_l=g_r;
      theta_r=theta;
      g_r=gg;
    endif;
    try = (abs(cr-gg) .< .01) +(gg>cr);
    if try==1;
      nogo=0; break;
    endif;
  endfor;
  if nogo==1; goto _end; endif;
  try= (abs(cr-g_r) .< .01)+(abs(theta_l-theta_r) .< .000001);
  if try==1; goto _end; endif;
  a1=a0; nogo=1;
  for j (1,20,1);
    theta=(theta_l+theta_r)/2;
    {g,a1,ret}=profile_gmm(theta,a1,iter);
    if ret>2;
      {g,a1,ret}=profile_gmm(theta,a0,iter);
    endif;
    if g>cr;
      theta_r=theta; g_r=g;
    else;
      theta_l=theta; g_l=g;
    endif;
    try= (abs(cr-g) .< .01)+(abs(theta_l-theta_r) .< .000001);
    if try==1; nogo=0; break; endif;
    theta=(theta_l*(g_r-cr)+theta_r*(cr-g_l))/(g_r-g_l);
    {g,a1,ret}=profile_gmm(theta,a1,iter);
    if ret>2;
      {g,a1,ret}=profile_gmm(theta,a0,iter);
    endif;
    if g>cr;
      theta_r=theta; g_r=g;
    else;
      theta_l=theta; g_l=g;
    endif;
    try= (abs(cr-g) .< .01)+(abs(theta_l-theta_r) .< .000001);
    if try==1; nogo=0; break; endif;
  endfor;
  _end:
retp(a1,theta,nogo);
endp;


/***********************************/
