/* GMM.PRG 

written by:

Bruce E. Hansen
Department of Economics
Social Science Building
University of Wisconsin
Madison, WI 53706-1393
bhansen@ssc.wisc.edu
http://www.ssc.wisc.edu/~bhansen/


This Gauss program calculates the GMM test for a nonlinear
restriction on the parameters of a linear regression, as described 
in Newey and West (IER, 1987) and discussed in
B. Hansen, "Edgeworth expansions for the Wald and GMM
Statistics for Nonlinear Restrictions", working paper (2000).

The model is a linear regression
Y=Xb + e
with independent observations.  Y is nx1, X is nxk and b is kx1.

The hypothesis to test is a non-linear function of the regression parameters
H0: g(b)=0

The end product are unrestricted and restricted estimates of b, and the
GMM test statistic for H0.

The user must provide:
  -  The data matrices: Y and X
      Y is an nx1 vector
       X is an nxk matrix.  (X should include an intercept if desired.)
  -  A procedure "g_contraint" to calculate   g(b)
     (This procedure should return a real number)
  -  A procedure "g_derivative" to calculate   d/db g(b)
     (This procedure should return a 1xk vector)
     If analytical derivatives are unable, you can use numerical derivatives.
     (An example is given below.)
     
There are also controls for
  -  White-Robust or Homoskedastic Covariance Matrix
  -  Name for the output file

The procedure requires the GAUSS module "Constrained Optimization".
Before running the procedure, make sure the CO library is in memory.
To do this, you can type at the Gauss prompt: "library co".

Below, we provide an example for the data matrices and the two procedures.
The example replicates the model of Gregory and Veal (1985, Econometrica)
for one parameterization.

***********************************/

/**** Data in Y and X matrices *****/
@ example @
load dat[100,4]=gmm.dat;
y=dat[.,1];
x=dat[.,2:4];

/******** Proc for g(beta) ********/

proc g_contraint(b);  
@ example @
local t;
  t=b[1]-1/b[2];
retp(t);
endp;

/******** Proc for d/db g(beta) ********/

proc g_derivative(b);  
@ example @
local d;
  d=1~(1/b[2]^2)~0;
retp(d);
endp;

/*  
Numerical Derivative:  Use if analytic derivatives unavailable:
proc g_derivative(b);  
local d;
  d=gradp(&g_constraint(b),b);
retp(d);
endp;
*/



/****************** Controls ************************/

_homo = 0;
	@ Set _homo=0 for Eicker-White covariance matrix
	  Set _homo=1 for "OLS" covariance matrix @
output file = "gmm.out" reset; output off;




/****************************************************/

coset;
string _co_Options = {stepbt newton forward none};
			@  This can be changed. In particular, the choice
			      of step length method may make a difference @
_co_MaxIters=500;    		@  This can be changed if desired @
trap 4;
_co_EqProc=&g_contraint;
_co_EqJacobian=&g_derivative;
_co_GradProc=&qmm_grad;
_co_HessProc=&qmm_hess;

/**********************************************************/

n=rows(y);cx=cols(x);
xx=moment(x,0);
xxi=invpd(xx);
xy=x'y;
betahat=xxi*xy;
e=y-x*betahat;
sig=(e'e)/(n-cx);
if _homo==1;
  om=xx*sig;
else;
  om=moment(x.*e,0);
endif;
v=xxi*om*xxi;
vi=invpd(v);
omi=invpd(om);
xxy=xx*omi*xy;
se=sqrt(diag(v));
_names=0 $+ "Alpha" $+ ftocv(seqa(1,1,cx),1,0);

{beta0,gm0,gr,ret0}=co(&qmm,betahat);
e=y-x*beta0;
sig1=(e'e)/(n-cx);
if _homo==1;
  om=xx*sig1;
else;
  om=moment(x.*e,0);
endif;
v=xxi*om*xxi;
g=g_derivative(beta0)';
gv=g'v;
v0=v-(gv'gv)/(gv*g);
se0=sqrt(diag(v0));

if _homo==1;
  gm1=gm0*sig/sig1;
else;
  omi=invpd(om);
  vi=xx*omi*xx;
  xxy=xx*omi*xy;
  {beta1,gm1,gr,ret1}=co(&qmm,beta0);
  v=xxi*moment(x.*(y-x*beta1),0)*xxi;
  g=g_derivative(beta1)';
  gv=g'v;
  v1=v-(gv'gv)/(gv*g);
  se1=sqrt(diag(v1));
endif;

/*********************************/

let fmt[1,3]="*.*lf" 16 8;
output on;
"CO exit code for First Optimization " ret0;
if _homo==0;
"CO exit code for Second Optimization " ret1;
endif;
"(Standard exit code is '0'.)";
"";"";
if _homo==1;
"Homoskedastic Covariance Matrix Used";
else;
"Eicker-White Covariance Matrix Used";
endif;
"";
"OLS Estimates of Linear Model";"";
pr=printfm("Variable"~"Estimate"~"St Error",0~0~0,fmt);"";
pr=printfm(_names~betahat~se,0~1~1,fmt);"";"";
"One-Step Constrained GMM Estimates";
"";
pr=printfm("Variable"~"Estimate"~"St Error",0~0~0,fmt);"";
pr=printfm(_names~beta0~se0,0~1~1,fmt);"";"";
if _homo==0;
"Two-Step Constrained GMM Estimates";
"";
pr=printfm("Variable"~"Estimate"~"St Error",0~0~0,fmt);"";
pr=printfm(_names~beta1~se1,0~1~1,fmt);"";"";
endif;
"One-Step GMM Test Statistic = " gm0;
"Asymptotic P-value = " cdfchic(gm0,1);
"";
"Two-Step GMM Test Statistic = " gm1;
"Asymptotic P-value = " cdfchic(gm1,1);
"";
output off;

/***********************************/

proc qmm(b);
local xe,xyv,xxv,omiv;
  xyv=varget("xy");
  xxv=varget("xx");
  omiv=varget("omi");
  xe=xyv-xxv*b;
retp(xe'omiv*xe);
endp;

proc qmm_grad(b);
local xxyv,vvi;
  xxyv=varget("xxy");
  vvi=varget("vi");
retp(-(xxyv-vvi*b)*2);
endp;

proc qmm_hess(b);
  local vvi;
  vvi=varget("vi");
retp(vvi*2);
endp;


/***********************************/
