%*****************************************************************
% --------ELIKE--------- 

%This is a MATLAB program file, written by
%Bruce E. Hansen
%Department of Economics
%University of Wisconsin
%http://www.ssc.wisc.edu/~bhansen/

%This version: 2/7/2003

%This procedure calculates empirical likelihood probabilities and parameter estimates.
%The program is self-contained, and does not require special applications.

%The main procedures are ELIKE_OPT and LAMBDA_OPT.
%These will calculate the outer and inner loops of EL estimation.

%For linear models, the program is very fast. 

%*****************************************************************

%Format:
%{beta,se,lambda,lik,p}=elike_opt(beta0);

%Inputs:
%beta0		kx1	starting value for parameter vector
%			A good choice is the GMM estimator
%Output:
%beta		kx1	parameter estimates
%se		kx1	asymptotic standard errors
%lambda		mx1	Lagrange multipliers
%lik		1x1	Empirical likelihood
%p		nx1	probabilities


%n is the sample size
%k is the number of parameters
%m is the number of moment conditions			

%*****************************************************************

%ELIKE_OPT requires four procedures which take as inputs the kx1 parameter
%vector beta, and the mx1 Lagrange multiplier vector lambda.

%These procedures are:

%(1)  e=moment_(beta);
%The nxm moment matrix

%(2)  d=Dstar_(beta,lambda);
%d is an	nxk matrix
%Each row is the derivative of the moment condition, 
%multiplied by lambda, i.e.
%   d/db moment_(b)' * lambda

%(3)  dd=DSum_(beta,lambda);
%dd is kxm, the sum of the derivative of the moment condition,
%divided by (1+_moment(beta)*lambda).

%(4)  ddd=_DHessian(beta,lambda);
%ddd is kxk, the sum of the second derivative of the moment condition 
%multiplied by lambda, divided by (1+_moment(beta)*lambda).
%For the linear model this is the zero matrix.


%********************************************************************

%The procedure ELIKE_OPT maximizes the profile empirical likelihood
%function using a modified Newton algorithm.  
%(The outer-loop of the EL problem.)

%The maximum number of iterations and tolerance for convergence
%are controlled by the first two lines of the procedure.
%These can be changed if desired.

%There are also controls for printing iteration output to the screen,
%and final estimates to an output file.  The output file is by dafault
%named "elike.out", and this can be changed.  Names can be provided
%for the parameters if desired, else by default they are labeled
%"Beta1", "Beta2". etc.

%Analytic gradients and a numerical Hessian is used.  
%At each iteration, the eigenvalues of the Hessian are examined, and
%eigenvalues smaller than a minimum (default 0.1) are replaced.

%The steplength is picked as follows.
%The steplenghts 1, 1/2, 1/4, 1/8, 1/16 are examined sequentially.
%For the first which produces an improvement in the likelihood, a quadratic
%approximation is calculated for an optimal steplength.  If none of these
%values produces an improvement in the likelihood, the steplength is set
%to 1/16.  Even though this reduces the likelihood, it is important to iterate
%the parameter vector to reinitiate the optimization.

%ELIKE_OPT calls the procedure LAMBDA_OPT (see below) which calculates the 
%LaGrange multiplier for the EL problem, the so-called inner loop.  

%The current iteration of the parameter vector beta is stored in the
%global variable "_beta", which is useful in case the iterations are
%stopped.

%********************************************************************
%Here we list the four necessary procedures, set up for the linear model 

%y = z*beta + e
%E(e|x)=0

%The dependent variable vector is y
%The RHS variable matrix is z
%The instrument matrix is x

%For this linear model, a good choice for starting value (beta0) is a GMM estimator. 
%The program "gmm.prc" calculates the GMM estimator for this linear model.

%********************************************************************/

function main()
% ***************Global Variables******************************* %
global MaxIters_;
MaxIters_=2000; 		% maximum number of newton iterations 	@
global MaxTol;MaxTol_ = .00001;		% tolerance for convergence		@
global output_;output_=1;			    % set to 1 for screen iteration output 	@
global print_;print_=1;			    % set to 1 to print estimates 	 	@
global out;out= fopen('elikeout.txt','wt');	% Name of output file 			@
global bnames_;bnames_=NaN;		% kx1 vector of names for parameters	@
global x;
global y;
global z;
load el.txt;
global y; y=el(:,1);
global z; z=el(:,2:3);
global x; x=el(:,4:7);
beta0=gmm_est(y,z,x);
% If set to miss(0,0), the program will substitute "Beta1" etc @
%declare beta_; 
elike_opt(beta0)
fclose (out);


function e=moment_(beta);
global x;
global y;
global z;

e=x.*((y-z*beta)*ones(1,length(x(1,:))));

function e=Dstar_(beta,lambda);
global x;
global y;
global z;
e=-z.*((x*lambda)*ones(1,length(z(1,:))));

function e=DSum_(beta,lambda);
global x;
global y;
global z;
e=-z'*(x./((1+moment_(beta)*lambda)*ones(1,length(x(1,:)))));

function h=DHessian_(beta,lambda);
k=length(beta(:,1));
h=zeros(k,k);

function beta=gmm_est(y,z,x);
xx=x'*x;
xxi=inv(xx);
xz=x'*z;
xy=x'*y;
beta=((xz'*xxi*xy)'/(xz'*xxi*xz)')';
size((y-z*beta))
u=x.*((y-z*beta)*ones(1,length(x(1,:))));
w=inv((u-ones(length(u(:,1)),1)*mean(u))'*(u-ones(length(u(:,1)),1)*mean(u)));
beta=((xz'*w*xy)'/(xz'*w*xz)')';
u=x.*((y-z*beta)*ones(1,length(x(1,:))));
w=inv((u-ones(length(u(:,1)),1)*mean(u))'*(u-ones(length(u(:,1)),1)*mean(u)));
beta=((xz'*w*xy)'/(xz'*w*xz)')';


% ************************************************************* %  
function [beta,se,lambda,lik,p]=elike_opt(beta);
global MaxIters_;
global MaxTol_;
global output_;
global print_;
global outfile_;
global bnames_;
global out;
%t1=hsec;
MaxStepIters=4;  % maximum number of steplength steps
MinEig=.1;       % minimum eigenvalue of Hessian
ret=1;
e=moment_(beta);
[lambda,lik,p,ret]=lambda_opt(e,0);
ddstar=Dstar_(beta,lambda);
gg=sum(ddstar./((1+e*lambda)*ones(1,length(ddstar(1,:)))))';
for it=1:MaxStepIters
    ddstar=Dstar_(beta,lambda);
    d=ddstar./((1+e*lambda)*ones(1,length(ddstar(1,:))));
    u=e./((1+e*lambda)*ones(1,length(e(1,:))));
    d2=u'*d-DSum_(beta,lambda)';
    uu=u'*u;
    warning off;
    hd=(d2'/uu')';
    mw=' ';
    [mw,idw] = lastwarn;
    warning on;
    lastwarn(' ');
    if 1-(mw==' ')
        hd=pinv(uu)*d2;
    end;
    h=d2'*hd-d'*d+DHessian_(beta,lambda);
    [eh,ev]=eig(h);
    ev=diag(ev);
    iv=1./(ev.*(ev>MinEig)+MinEig.*(ev<=MinEig));
    direc=(eh*((iv*ones(1,length(eh(:,1)))).*(eh')))*gg;
    step=(2^(-MaxStepIters));
    beta1=beta-direc*step;
    [lambda0,lik0,p,ret0]=lambda_opt(moment_(beta1),0);
    liikkkk=lik0*(10^14)-floor(lik0*(10^14))
    if lik0>lik;
        beta=beta1;
        lik=lik0;
        lambda=lambda0;
    else
        step=1;
        for si=1:(MaxStepIters+1)
            steps=si;
            step1=step;
            beta1=beta-direc*step1;
            e1=moment_(beta1);
            [lambda1,lik1,p,ret1]=lambda_opt(e1,0);
            if lik1<lik
                step2=step/2;
                beta2=beta-direc*step2;
                e2=moment_(beta2);
                [lambda2,lik2,p,ret2]=lambda_opt(e2,0);
                break;
            end;
            step=step/2;
        end;
        step3=(lik1+3*lik-4*lik2)/(4*lik1+4*lik-8*lik2);
        beta3=beta-direc*step3;
        e3=moment_(beta3);
        [lambda3,lik3,p,ret3]=lambda_opt(e3,0);
        [lik,si]=min([lik1;lik2;lik3]);
        step=step1*(si==1)+step2*(si==2)+step3*(si==3);    
        beta=beta1*(si==1)+beta2*(si==2)+beta2*(si==3);
        e=e1.*(si==1)+e2.*(si==2)+e3.*(si==3);
        lambda=lambda1*(si==1)+lambda2*(si==2)+lambda3*(si==3);
    end;
    ddstar=Dstar_(beta,lambda);
    gg=sum(Dstar_(beta,lambda)./((1+e*lambda)*ones(1,length(ddstar(1,:)))))';
    if output_==1;
        fprintf(out,'=======================\n');
        fprintf(out,'Iteration %u\n',it);
        fprintf(out,'Steplength %f\n',step);
        fprintf(out,'Function %f\n',lik);
        fprintf(out,'-----------------------\n');
        fprintf(out,'  Parameter          Value         Gradient \n');
        for ii=1:length(beta(:,1))
             fprintf(out,'%f    %f    %f\n',ii,beta(ii),gg(ii));
        end;
        beta_=beta;
        if (max(abs(gg'/lik'))<MaxTol_);
            ret=0;
            break;
        end;
    end;
end;
d=DSum_(beta,lambda);

v=inv(d*inv((e./((1+e*lambda)*ones(1,length(e(1,:)))))'*(e./((1+e*lambda)*ones(1,length(e(1,:))))))*d');
se=sqrt(diag(v));
n=length(e(:,1));
q=length(lambda(:,1))-length(beta(:,1));
%t2=hsec;
if print_==1;
    if isnan(bnames_(1))==1
        names_=strcat('Beta',num2str(1),'_');
        for ii=2:length(beta(:,1));
            if ii<10
                names_=[names_;strcat('Beta',num2str(ii),'_')];
            else
                names_=[names_;strcat('Beta',num2str(ii))];
            end;
        end;
    else
        names_=bnames_;
    end;
    fprintf(out,'\n\n Empirical Likelihood Estimation\n');
    fprintf(out,'Sample Size  %u\n',n);
    fprintf(out,'Number of Instruments  %u\n',length(lambda(:,1)));
    fprintf(out,'Overidentifying Restrictions  %f\n',q);
    fprintf(out,'Log Empirical Likelihood  %f\n',-n*log(n)-lik);
    fprintf(out,'Test of Over-Identifying Restrictions  %f\n',lik);
    rchi=1-chi2cdf(lik,q);
    if isnan(rchi)
        rchi=-1;
    end;
    fprintf(out,'Asymptotic P-Value  %f\n',rchi);
    %fprintf(out,'Computation Time (Seconds)',);
    if ret==1;
        fprintf(out,'Maximum Iterations Reached before Convergence\n\n');
    end;
    fprintf(out,'  Variable     Estimate     St Error\n');
    for jj=1:length(beta(:,1))
        fprintf(out,'%s     %f   %f\n',names_(jj,:),beta(jj),se(jj));
    end;
    fprintf(out,'\n\n');
end;

%*****************************************************************
%ELIKE_OPT calls the procedure LAMBDA_OPT which calculates the 
%LaGrange multiplier for the EL problem, the so-called inner loop.  
%This program is self-contained and does not rely on the GMM structure 
%in any way.  It could be used in an alternative context to calculate 
%EL probabilities, for instance.

%The optimization method is similar to ELIKE_OPT, with one
%difference that at each iteration step, the program checks for the positivity 
%of the probability vector.  If the constraint is not satisfied, a smaller 
%stepsize is calculated.

%When ELIKE_OPT is run, the iteration steps for LAMBDA_OPT are not printed
%to the screen.

%The procedure LAMBDA_OPT has the format 
%[lambda,f,p,ret]=lambda_opt(e,output);

%Input:
%e		nxm 	moment conditions
%output		1x1	1 if iteration output printed to screen

%Output:
%lambda		mx1	final value
%f		1x1	log-likelihood
%p		nx1	probability vector
%ret		1x1	return code
%			ret=0 for normal convergence
%			ret=1 if maximum iterations reached
%			ret=2 if failure to find a stepsize
%********************************************************************/

function [mlambda,nlik,p,ret]=lambda_opt(e,output);
global output_;
global print_;
global outfile_;
global bnames;
global out;

MaxIters=100; 		% maximum number of newton iterations 	@
MaxStepIters=13;		% maximum number of steplength steps  	@
MaxTol = .00001;		% tolerance for convergence		@
MinEig =.1;			% minimum eigenvalue of Hessian		@
n=length(e(:,1));
n1=1/n;
lambda=zeros(length(e(1,:)),1);
w=ones(n,1);
ret=0;
lik=0;
uu=e'*e;
uu
chol(uu)
warning off;
m=inv(chol(uu));
mw=' ';
[mw,idw] = lastwarn;
warning on;
lastwarn(' ');
if 1-(mw==' ')
    [eh,ev]=eig(h);
    ev=diag(ev);
    iv=(ev.*(ev > MinEig) + MinEig .* (ev <= MinEig));
    m=eh.*(1./sqrt(iv));
end;
em=e*m;
for it=1:MaxIters
    w*(ones(1,length(em(1,:))))
    u=em./(w*(ones(1,length(em(1,:)))));
    uu=u'*u;
    su=sum(u)';
    warning off;
    direc=(su'/uu')';
    mw=' ';
    [mw,idw] = lastwarn;
    warning on;
    lastwarn(' ');
    if 1-(mw==' ')
        disp('lxbdlxbdlxbdlxbdlxbdlxbd');
        [eh,ev]=eig(h);
         ev=diag(ev);
        iv=1./(ev.*(ev > mineig) + mineig .* (ev <= mineig));
        direc=(eh*(iv.*(eh')))*su;
    end;
    step=1;
    for si=1:MaxStepIters
        steps=si;
        lambda1=lambda+direc*step;
        w=1+em*lambda1;
        if min(w)>n1
            lik1=-sum(log(w))';
            if lik1<lik;
                lambda2=lambda+direc*step/2;
                lik2=-sum(log(1+em*lambda2));
                break;
            end;
        end;
        step=step/2;
    end;
    if steps==MaxStepIters
        ret=2;
        break;
    end;
    step=(lik1+3*lik-4*lik2)/(4*lik1+4*lik-8*lik2);
    lambda3=lambda+direc*step;
    if (min(1+em*lambda3)>n1)
        lik3=-sum(log(1+em*lambda3))';
        [temp,si]=min([lik1;lik2;lik3]);
        lambda=lambda1*(si==1)+lambda2*(si==2)+lambda3*(si==3);
    else
        [temp,si]=min([lik1;lik2]);
        lambda=lambda1*(si==1)+lambda2*(si==2);
    end;
    w=1+em*lambda;
    lik=-sum(log(w))';
    test=max(abs(em'*(1./w)))/n;
    if output_==1
        fprintf(out,'Iteration %u\n',it);
        fprintf(out,'Steps     %u\n',steps);
        fprintf(out,'Steplength %u\n',step);
        fprintf(out,'Function %f\n',lik);
        fprintf(out,'Max Error %f\n',test);
        fprintf(out,'-----------------------\n');
        fprintf(out,'  Parameter          Value\n');
        mlambda=m*lambda;
        for jj=1:length(lambda(:,1))
            fprintf(out,'%u    %f\n',jj,mlambda(jj));
        end;
        fprintf(out,'\n\n');
    end;
    if steps==MaxStepIters;
        ret=2;
        break;
    end;
    if test<MaxTol
        break;
    end;
    if it==MaxIters
        ret=1;
    end;
end;
p=1./((1+em*lambda)*n);
p=p/sum(p)';
mlambda=m*lambda;
nlik=-lik;%?????????????????

        
        
            
    