/********************************************************
CDF.PRC

written by
Bruce E. Hansen
Department of Economics
University of Wisconsin
www.ssc.wisc.edu/~bhansen

This Gauss program implements the methods reported in my paper
"Nonparametric Estimation of Smooth Conditional Distributions".

This version: 5/27/2004

There are three procedures given here: LL_CDF, SLL_CDF, and MSE_TILDE

*****************************************************************

LL_CDF

Estimates the conditional distribution function of Y given X, F(s|t)
using a Local Linear Estimator

The bandwidth is selected by an empirical plug-in rule.

format:
{cdf,b}=LL_CDF(y,x,s,t)

inputs:
y	Dependent variable 			nx1
x	Regressor				nx1
s	points of evaluation for y		rx1
t	points of evaluation for x		1x1

outputs:
cdf	Conditional Distribution Function	kx1
b	Bandwidth 				1x1

*****************************************************************

SLL_CDF

Estimates the conditional distribution function of Y given X, F(s|t)
using a Smoothed Local Linear Estimator

The bandwidth is selected by an empirical plug-in rule.  The function
MSE_TILDE (see below) is minimizes over possible bandwidths. 
The internal GAUSS routine QNEWTON is used to minimize the function.

format:
{cdf,h,b}=SLL_CDF(y,x,s,t)

inputs:
y	Dependent variable 			nx1
x	Regressor				nx1
s	points of evaluation for y		rx1
t	points of evaluation for x		1x1

outputs:
cdf	Conditional Distribution Function	kx1
h	Bandwidth for y				1x1
b	Bandwidth for x				1x1

*****************************************************************

MSE_TILDE

Estimates the asymptotic MSE of the SLL estimator as a function of transformed
bandwidths.  This procedure is called and minimized by SLL_CDF to implement 
the bandwidth selection rule.  It relies on the globals
_ghat,_vtilde,_v1tilde,_V2hat,_v3hat,_ht,_n,__output
which are declared in the first line below.  The bandwidths are transformed to
impose positivity and bounding constraints.

******************************************************************/

declare _ghat,_vtilde,_v1tilde,_V2hat,_v3hat,_ht,_n,__output;

proc (2) = LL_CDF(y,x,s,t);
local n,p,xt,sx,shat,that,rhat,ghat1,xtt,g2hat,rho,ghat,x0,xp,x1,x3,xpp,yp,v,vhat,
nn,y1,y2,ym,v1hat,v4hat,q0,w0,m1,vtilde,q1,w1,m3,v1tilde,b,beta,w,cdf;

  n=rows(y);
  p=5;		@ polynomial order for parametric approx @
  xt=x-t;  
  sx=stdc(x);
  shat=1.06*sx/(n^(.2));
  that=.94*sx/(n^(1/9));
  ghat1=meanc(pdfn(xt/shat))/shat;
  xtt=xt/that;
  g2hat=meanc(pdfn(xtt.*(that.^2-1)))/(that.^3);
  rhat=(ghat1/2/sqrt(pi)/(g2hat^2)/n).^(1/9);
  ghat=meanc(pdfn(xt/rhat))/rhat;

  x0=ones(n,1)/n;
  xp=xt.^(seqa(0,1,p+1)');
  x1=xp[.,1 2];
  x3=xp[.,1:4];

  trap 1;
  xpp=xp*invpd(moment(xp,0));
  if scalerr(xpp);
    xpp=xp*pinv(moment(xp,0));
  endif;
  trap 0;
  yp=y-y';
  yp=yp.*(yp.>0);
  v=xpp'yp*xpp;
  vhat=abs(v[1,1]);

  nn=ones(n,1);
  y1=vecr(y*(nn'))';
  y2=vecr(nn*(y'))';
  ym=reshape(maxc(y1|y2),n,n);
  v=xpp'ym*xpp;
  v1hat=abs(v[3,3]);
  v=xpp'ym*xpp;
  v4hat=abs(v[5,5]);

  q0=.78*((vhat/ghat/v1hat/n).^(.2));
  w0=pdfn(xt/q0);
  trap 1;
  m1=invpd(x1'(w0.*x1));
  if scalerr(m1);  m1=pinv(x1'(w0.*x1)); endif;
  trap 0;
  x1=(w0.*x1)*m1;
  v=x1'yp*x1;
  vtilde=abs(v[1,1]);

  q1=1.01*((vhat/ghat/v4hat/n).^(1/9));
  w1=pdfn(xt/q1);
  trap 1;
  m3=invpd(x3'(w1.*x3));
  if scalerr(m3);  m1=pinv(x3'(w1.*x3)); endif;
  trap 0;
  x3=(w1.*x3)*m3;
  v=x3'ym*x3;
  v1tilde=abs(v[3,3]);

  @ Bandwidth for Local Linear Estimation @
  b=.78*((vtilde/ghat/v1tilde/n).^(.2));
  w=pdfn(xt./b);
  beta=((xt'w)/((xt.^2)'w));
  w=w.*(1-xt*beta).*(xt*beta.<1);
  w=w./sumc(w);
  cdf=(y .<= s')'w; 
retp(cdf,b);
endp;  




proc (3) = SLL_CDF(y,x,s,t);
local n,p,xt,sx,shat,that,ghat1,xtt,g2hat,rhat,x0,xp,x1,x3,xpp,yy,yp,
v,vhat,nn,y1,y2,ym,v1hat,v4hat,q0,w0,m1,q1,w1,m3,j,r,gc,rc,h1,
m,a,u,h0,i,hr,phi,phi2,btilde,bh,h,b,w,beta,c,er,k,cdf,pdf;

  n=rows(y);_n=n;
  p=5;		@ polynomial order for parametric approx @
  xt=x-t;  
  sx=stdc(x);
  shat=1.06*sx/(n^(.2));
  that=.94*sx/(n^(1/9));
  ghat1=meanc(pdfn(xt/shat))/shat;
  xtt=xt/that;
  g2hat=meanc(pdfn(xtt.*(that.^2-1)))/(that.^3);
  rhat=(ghat1/2/sqrt(pi)/(g2hat^2)/n).^(1/9);
  _ghat=meanc(pdfn(xt/rhat))/rhat;

  x0=ones(n,1)/n;
  xp=xt.^(seqa(0,1,p+1)');
  x1=xp[.,1 2];
  x3=xp[.,1:4];

  trap 1;
  xpp=xp*invpd(moment(xp,0));
  if scalerr(xpp);
    xpp=xp*pinv(moment(xp,0));
  endif;
  trap 0;
  yy=y-y';
  yp=yy.*(yy.>0);
  v=xpp'yp*xpp;
  vhat=abs(v[1,1]);

  nn=ones(n,1);
  y1=vecr(y*(nn'))';
  y2=vecr(nn*(y'))';
  ym=reshape(maxc(y1|y2),n,n);
  v=xpp'ym*xpp;
  v1hat=abs(v[3,3]);
  v=xpp'ym*xpp;
  v4hat=abs(v[5,5]);

  q0=.78*((vhat/_ghat/v1hat/n).^(.2));
  w0=pdfn(xt/q0);
  trap 1;
  m1=invpd(x1'(w0.*x1));
  if scalerr(m1);  m1=pinv(x1'(w0.*x1)); endif;
  trap 0;
  x1=(w0.*x1)*m1;
  v=x1'yp*x1;
  _vtilde=abs(v[1,1]);

  q1=1.01*((vhat/_ghat/v4hat/n).^(1/9));
  w1=pdfn(xt/q1);
  trap 1;
  m3=invpd(x3'(w1.*x3));
  if scalerr(m3);  m1=pinv(x3'(w1.*x3)); endif;
  trap 0;
  x3=(w1.*x3)*m3;
  v=x3'ym*x3;
  _v1tilde=abs(v[3,3]);

  @ DF Bandwidth Estimation -- from Hansen (2004) @
  j=4;
  r=gamma(j+1.5)/2/pi/(stdc(y).^(2*j+3));
  for m (j,1,-1);
    a=(gamma(m+.5)*(2^(m+.5))/pi/r/n)^(1/(2*m+3));
    u=yy/a;
    h0=1;h1=u;
    for i (1,2*m-1,1);
      hr=u.*h1-i*h0;
      h0=h1; h1=hr;
    endfor;
    r=abs(meanc(meanc(pdfn(u).*hr)))/(a^(1+2*m));
  endfor;
  a=1/((sqrt(pi)*r*n)^(1/3));

  phi=pdfn(yy/a)/a;
  v=xpp'phi*xpp;
  _v2hat=v[3,1];

  phi2=phi.*((yy/a).^2 - 1)/(a^2);
  v=xpp'phi2*xpp;
  _v3hat=abs(v[1,1]);

  btilde=.78*((_vtilde/_ghat/_v1tilde/n).^(.2));
  bh=ln(btilde)|0;
  _ht=sqrt(pi)*_vtilde/2;
  __output=0;
  {h,m,gc,rc}=qnewton(&mse_tilde,bh);
  b=exp(h[1]);
  h=(_ht/(1+exp(-h[2])));

  w=pdfn(xt./b);
  beta=((xt'w)/((xt.^2)'w));
  w=w.*(1-xt*beta).*(xt*beta.<1);
  w=w./sumc(w);

  @ Numerical Approximation to Normal CDF @
  u=(s'-y)./h;
  let c = 1.330274429 1.821255978 1.781477937 .356563782 .31938153;
  t=1./(1+abs(u)*.2316419);
  er = 1 - ((((c[1].*t-c[2]).*t+c[3]).*t-c[4]).*t+c[5]).*t.*pdfn(u);
  k=(u.==0).*(.5)+(u.>0).*er+(u.<0).*(1-er);
  cdf=k'w; 
  
retp(cdf,h,b);
endp;  
 




proc mse_tilde(hh);
local b,h,sqrtpi,m1,mh;
  b=exp(hh[1]);
  h=_ht/(1+exp(-hh[2]));
  sqrtpi=sqrt(pi);
  m1=(b.^4)*_v1tilde-2*(h.^2).*(b.^2).*_v2hat+(h.^4).*_v3hat;
  mh=(_vtilde-h/sqrtpi)/(2*sqrtpi*_ghat*_n*b)+m1/4;
retp(mh);
endp;



