function F = Farebrother(x,lambda,tolerance,maxit)

% 	Weighted chi-square CDF calculated by Farebrother's algorithm
%	Written by Bruce E. Hansen 

% 	Arguments:
%	x		scalar, point at which distribution is to be evaluatedn
% 	lambda		vector, the weights of the weighted chi-square distribution
%			i.e., a set of non-zero eigenvalues
%	tolerance	scalar, desired level of accuracy for distribution function
%			default (0.001) set below
% 	maxit		scalar, the maximum number of terms in series approximation
%			default (50000) set below
%			
%
% 	Output:
%	F		scalar, cdf evalualted at x
% 			If convergence fails, F=missing
%			Computes P[Q<=x] where Q is weighted sum of chi-square(1) random variables
%
%			F is a weighted sum of chi-square CDFs
%			Expression is due to Ruben (1962) and Farebrother (1984)

% Set defaults
if nargin < 3
	tolerance = .001;	
end
if nargin < 4
	maxit = 50000;
end

beta = min(lambda); 
K = length(lambda);
K2 = K/2;
if K==2
	betalambda = min(lambda)./max(lambda);
else
	betalambda = beta./lambda;
end
bm = sqrt(prod(betalambda));
F = missing;

if bm > eps
	bs = bm;
	b = bm;
	am = 1/2;
	xbeta = x/beta;
	Gm = chi2cdf(xbeta,K);
	xbeta = xbeta/2;
	G = Gm*bm;
	gm = - xbeta + (K2-1)*log(xbeta) - gammaln(K2);
	for m = 1:maxit
		if K==2
			bm = bm*(1-betalambda)*(m+1/2)/(m+1);
		else
			am = (1-betalambda).*am;
			a(m,1) = sum(am);
			bm = (b*flip(a))/m;
			b(m+1) = bm;
		end
		bs = bs + bm;
		gm = gm + log(xbeta/(K2+m-1));
		Gm = Gm - exp(gm);
		G = G + Gm*bm;
		Tm = (1-bs)*Gm;
		if (Tm <= tolerance)
			F = G;
		break
	end
end
end
